#include <windows.h>
#include <math.h>
#include <stdarg.h>
#include "piemenu.h"
/* ------------------------------------------------------------------------
     Functions for creating, drawing, and operating Pie Menus:
      CreatePieMenuIndirect()  - creates a Pie Menu data structure
      DrawPieMenu()            - draws the Pie Menu given a Device Context
      PieMenuProc()            - determines what option was selected
      GetViewportWindowParms() - gets Device Context parameters for Menu
      DP_to_LP()               - converts cursor coordinates without a
                                 Device Context

                copyright (c) 1991, 1992 by Carl C. Rollo
   --------------------------------------------------------------------- */

BOOL CreatePieMenuIndirect (LPPIEMENUSTRUCT p, short *pwidth, short *pheight)
 /*
 ** Initializes a PIE_MENU structure. Several more time-consuming
 ** calculations are made in this routine to ensure that the pie menu is
 ** displayed with minimum delay. This routine must be called before a
 ** pie menu is displayed! The allocation of the PIE_MENU structure is left
 ** to code outside this routine so that the programmer may choose to store
 ** the structure in local heap, data segment, or global memory.
 **
 ** The routine returns TRUE if no error was encountered, FALSE otherwise.
 */
 {
  short i, j, xc, yc;
  TEXTMETRIC tm;
  HDC hdc, hdcMem;
  double incvalue, angle, halfangle, h, rsquare;
  DWORD dwsize;
  short tx[4], ty[4], halfwid, halfht;

  if ( (p != NULL) && (0 < p->n) && (p->n <= MAXOPTIONS) &&
       (pwidth != NULL) && (pheight != NULL) && (*pwidth > 0) &&
       (*pheight > 0) )
   {
    if (p->radius <= 0)
     if (p->InsideLabels)
      p->radius = min (*pwidth, *pheight)/2;
     else
      return (FALSE);  /* must have a radius for outside labels */

    rsquare = p->radius * p->radius;
    incvalue = TWO_PI / p->n;  /* value in radians of each wedge */
    halfangle = incvalue/2.0;

    /* Calculate the angle of the upper bound of each wedge
    ** and the endpoint coordinates of the radius at this angle.
    */
    for (i = 0; i < p->n; i++)
     {
      p->option[i].maxangle = incvalue * (i + 1);
      p->option[i].endarc.x = p->radius * cos (p->option[i].maxangle) + .5;
      p->option[i].endarc.y = p->radius * sin (p->option[i].maxangle) + .5;
     }

    /* Build brush handles needed to fill the pie menu. */
    for (i = 0; i < p->n; i++)
     p->option[i].brush = CreateSolidBrush(p->option[i].color);

    /* Determine the label positions for each wedge. */
    hdc = CreateIC ("DISPLAY", NULL, NULL, NULL);
    hdcMem = CreateCompatibleDC (hdc);
    SetMapMode (hdcMem, MM_ISOTROPIC);
    if (p->InsideLabels)
     SetWindowExt (hdcMem, p->radius, p->radius);
    else
     SetWindowExt (hdcMem, *pwidth/2, *pheight/2);
    SetViewportExt (hdcMem, *pwidth/2, -(*pheight)/2);
    SetViewportOrg (hdcMem, *pwidth/2, *pheight/2);

    SelectObject (hdcMem, GetStockObject(SYSTEM_FONT));
    GetTextMetrics (hdcMem, &tm);

    /* Determine label positions for each wedge. */
    if (p->InsideLabels)
     {
      /* Labels are to be positioned inside the pie diagram.
      ** Calculate the label position for each wedge.
      */
      h = 0.7 * (p->radius * cos (halfangle));
      for (i = 0; i < p->n; i++)
       {
	if (p->option[i].name == NULL)
	 {
	  p->option[i].showlabel = FALSE;
	  continue;
	 }

	/* Positioning varies with number of wedges. */
	switch (p->n)
	 {
	  case 1 :
		   xc = 0; yc = 0;
		   break;

	  case 2 : if (i == 0)
		    {
		     xc = 0; yc = p->radius/2;
		    }
		   else
		    {
		     xc = 0; yc = -(p->radius)/2;
		    }
		   break;

	   case 3 : angle = p->option[i].maxangle - halfangle;
		   xc = p->radius/2 * cos (angle);
		   yc = p->radius/2 * sin (angle);
		   break;

	  default :
		   angle = p->option[i].maxangle - halfangle;
		   xc = h * cos (angle);
		   yc = h * sin (angle);
	 }

	/* Determine the dimensions of the label string using the
	** SYSTEM_FONT. */
	dwsize = GetTextExtent (hdcMem, p->option[i].name,
	  lstrlen (p->option[i].name));

	/* Get coordinates of corners of the rectangle bounding the label */
	halfwid = (double) (LOWORD (dwsize))/2.0 + 0.5;
	halfht  = (double) (HIWORD (dwsize))/2.0 + 0.5;
	tx[0] = xc - halfwid;
	ty[0] = yc + halfht;
	tx[1] = xc + halfwid; ty[1] = ty[0];
	tx[2] = tx[1]; ty[2] = yc - halfht;
	tx[3] = tx[0]; ty[3] = ty[2];

	/* Calculate the counter-clockwise total angles for lines connecting
	** the center of the pie to each of the corners of the rectangle
	** bounding the label, and make sure that each line lies within
	** the wedge boundaries. */
	for (j = 0; j < 4; j++)
	 {
	  if ( (tx[j] * tx[j] + ty[j] * ty[j]) > rsquare)
	   break;   /* point is outside the pie */

	  if (tx[j] == 0)
	   if (ty[j] > 0)
	    angle = PI / 2.0;
	   else
	    angle = -PI / 2.0;
	  else
	   angle = atan ( (double) ty[j] / (double) tx[j] );

	  if ( (tx[j] >= 0) && (ty[j] < 0) )
	   angle += TWO_PI;
	  else
	   if (tx[j] < 0)
	    angle += PI;

	  if ( (angle <= ((i == 0) ? 0.0 : p->option[i-1].maxangle)) ||
	       (angle >= p->option[i].maxangle) )
	   break;

	 }

	/* If one of the corners lies outside the wedge, set switch to
	** indicate label is not to be shown.
	*/
	if (j < 4)
	 p->option[i].showlabel = FALSE;
	else
	 {
	  /* Set the location for the label */
	  p->option[i].txt.left   = tx[0];
	  p->option[i].txt.top    = ty[0];
	  p->option[i].txt.right  = tx[2];
	  p->option[i].txt.bottom = ty[2];
	  p->option[i].showlabel = TRUE;
	 }
       }
     }
    else
     {
      /* Labels are to be positioned outside the pie diagram. */

      for (i = 0; i < p->n; i++)
       {
	if (p->option[i].name == NULL)
	 {
	  p->option[i].showlabel = FALSE;
	  continue;
	 }

	/* Get the midpoint of the wedge. */
	if (i == 0) angle = 0.0;
	else angle = p->option[i-1].maxangle;

	angle = (p->option[i].maxangle - angle) / 2.0 + angle;

	xc = p->radius * cos (angle);
	yc = p->radius * sin (angle);

	/* Determine the dimensions of the label string using the
	** SYSTEM_FONT. */
	dwsize = GetTextExtent (hdcMem, p->option[i].name,
		 lstrlen (p->option[i].name));

	/* Position the text label based upon the quadrant in which the
	** wedge is located. */
	if (angle >= 0.0 && angle < PI/2.0)
	 {
	  /* First quadrant */
	  p->option[i].txt.left   = xc;
	  p->option[i].txt.top    = yc + HIWORD(dwsize);
	  p->option[i].txt.right  = xc + LOWORD(dwsize);
	  p->option[i].txt.bottom = yc;
	 }
	else if (angle >= PI/2.0 && angle < PI)
	 {
	  /* Second quadrant */
	  p->option[i].txt.left   = xc - (LOWORD(dwsize) + tm.tmAveCharWidth);
	  p->option[i].txt.top    = yc +  HIWORD(dwsize);
	  p->option[i].txt.right  = xc;
	  p->option[i].txt.bottom = yc;
	 }
	else if (angle >= PI && angle < 3.0 * PI/2.0)
	 {
	  /* Third quadrant */
	  p->option[i].txt.left   = xc - (LOWORD(dwsize) + tm.tmAveCharWidth);
	  p->option[i].txt.top    = yc;
	  p->option[i].txt.right  = xc;
	  p->option[i].txt.bottom = yc - HIWORD(dwsize);
	 }
	else
	 {
	  p->option[i].txt.left   = xc;
	  p->option[i].txt.top    = yc;
	  p->option[i].txt.right  = xc + (LOWORD(dwsize) + tm.tmAveCharWidth);
	  p->option[i].txt.bottom = yc -  HIWORD(dwsize);
	 }

	p->option[i].showlabel = TRUE;

       }
     }
    DeleteDC (hdc);
    DeleteDC (hdcMem);
    return (TRUE);
   }
  return (FALSE);
 }

/* ------------------------------------------------------------------ */

void DrawPieMenu (HDC hdc, LPPIEMENUSTRUCT p)
 /*
 ** Draws a Pie Menu on the Device Context pointed to by "hdc", using
 ** information given in the structure pointed to by "p".
 */

 {
  short i;
  POINT center;
  DWORD dwOrg;
  HFONT hCurrFont;

  if (p != NULL)
   {
    hCurrFont = SelectObject (hdc, GetStockObject (SYSTEM_FONT));
    SetTextColor (hdc, BLACK);

    for (i = 0; i < p->n; i++)
     {
      SelectObject (hdc, p->option[i].brush);

      Pie (hdc, -(p->radius), p->radius, p->radius, -(p->radius),
	  (i == 0) ? p->radius : p->option[i-1].endarc.x,
	  (i == 0) ? 0 : p->option[i-1].endarc.y,
	  p->option[i].endarc.x, p->option[i].endarc.y);

      if (p->option[i].showlabel)
       {
	if (p->option[i].color == BLACK)
	 SetBkColor (hdc, WHITE);
	else
	 SetBkColor (hdc, p->option[i].color);

	DrawText (hdc, p->option[i].name, -1, &(p->option[i].txt),
		  DT_CENTER);
       }

      if (p->UseChecks != NULL)
       if (p->option[i].checked)
	InvertRect (hdc, &(p->option[i].txt));
     }

    /* Center the cursor in the Pie Menu */
    dwOrg = GetDCOrg(hdc);
    center.x = 0; center.y = 0;
    LPtoDP (hdc, (LPPOINT) &center, 1);
    center.x += LOWORD (dwOrg); center.y += HIWORD (dwOrg);
    SetCursorPos (center.x, center.y);

    SelectObject (hdc, hCurrFont);
   }
 }

/*  --------------------------------------------------------------------- */

short PieMenuProc (LPVIEWWINDATA vwd, LPPIEMENUSTRUCT p)
 {
  /*
  ** Retrieves the position of the cursor from within the Pie Menu
  ** and converts this into a selection value. If the position cannot be
  ** converted, the "no_choice" value is returned.
  */

  short selection;
  short x, y, i, j;
  POINT position;
  double angle;
  RECT rect;

  /* Try to convert position coordinates into a selection */

  GetCursorPos( (LPPOINT) &position);

  position.x -= LOWORD(vwd->DCOrg); position.y -= HIWORD(vwd->DCOrg);
  DP_to_LP (vwd, (LPPOINT) &position, 1);

  x = position.x; y = position.y;
  selection = p->no_choice;     /* set result at "no selection" */

  if ( ((double)x * (double)x + (double)y * (double)y) >
     ((double)p->radius * (double)p->radius) )
   return (selection);     /* outside the circle */

  if (x == 0)
   if (y > 0)
    angle = PI/2.0;
   else
    angle = -PI/2.0;
  else
   angle = atan( (double) y / (double) x);

  if ((x >= 0) && (y < 0))
   angle += TWO_PI;
  else if (x < 0)
   angle += PI;

  for (i = 0; i < p->n; i++)
   if ( (angle >= ((i == 0) ? 0.0 : p->option[i-1].maxangle)) &&
	(angle < p->option[i].maxangle))
    {
     selection = p->option[i].value;
     break;
    }

  /* If we are using labels and inverting the previously selected
  ** option, shut the previously inverted label off here. */
  if (p->UseChecks)
   {
    for (j = 0; j < p->n; j++)
     if (p->option[j].checked)
      {
       p->option[j].checked = FALSE;
      }

    /* ...and turn the new one on here */
    if (selection != p->no_choice)
     p->option[i].checked = TRUE;
   }

  return(selection);

 }

/*  --------------------------------------------------------------------- */

 BOOL DP_to_LP (LPVIEWWINDATA vd, LPPOINT p, int ncount)
 /*
 ** A private version of the Windows SDK function, "DPtoLP", that does not
 ** use a Device Context to convert points from device coordinates to
 ** logical coordinates. DP_to_LP requires a pointer to a structure of type,
 ** VIEWWINDATA, that has been previously loaded with the window-viewport
 ** parameters required to convert the points. The remaining arguments are
 ** the same as those for "DPtoLP". The function returns TRUE if the points
 ** can be converted, FALSE otherwise.
 */
  {
   if (vd != NULL && ncount > 0)
    {
     while (ncount--)
      {
       p->x = (p->x - LOWORD((vd->ViewportOrg)) ) *
	(double) ((int) LOWORD((vd->WindowExt))) /
	(double) ((int) LOWORD((vd->ViewportExt)))
	+ LOWORD((vd->WindowOrg));

       p->y = (p->y - HIWORD((vd->ViewportOrg)) ) *
	(double) ((int) HIWORD((vd->WindowExt))) /
	(double) ((int) HIWORD((vd->ViewportExt)))
	+ HIWORD((vd->WindowOrg));
       p++;
      }
     return (TRUE);
    }
   else return (FALSE);
  }

/*  --------------------------------------------------------------------- */

 LPVIEWWINDATA GetViewportWindowParms (HDC hDC)
 /*
 ** Gets the origins and extents of the viewport and window definitions
 ** specified for the Device Context whose handle is passed in hDC. Also
 ** stores the origin of the Device Context. All results are stored in a
 ** structure of type, VIEWWINDATA, whose address is returned. A NULL
 ** handle passed to the routine returns a NULL pointer.
 */
  {
   static VIEWWINDATA vwd;

   if (hDC != NULL)
    {
     vwd.DCOrg       = GetDCOrg(hDC);
     vwd.ViewportExt = GetViewportExt(hDC);
     vwd.ViewportOrg = GetViewportOrg(hDC);
     vwd.WindowExt   = GetWindowExt(hDC);
     vwd.WindowOrg   = GetWindowOrg(hDC);
     return ( (LPVIEWWINDATA) &vwd);
    }
   else
    return (NULL);
  }

/*  --------------------------------------------------------------------- */
