/*

Sweep Test - Test AllocSeg.ASegSweep

*/

#include "array.h"
#include "aseg.h"
#include "bool.h"
#include "power2.h"
#include "stdio.h"

FILE*	TraceFile;

typedef enum BlockKindTg {
	FreeBlock,
	UnrefBlock,
	RefBlock,
	EndBlock
} BlockKind;
	
typedef struct BlockDescTg {
	BlockKind	Kind;
	unsigned	Offset;
	unsigned	Length;
} BlockDesc;

typedef BlockDesc	SegDesc[];

void CheckCondition(
	Boolean 		Passed,
	char*			TestName,
	char*			Cause
) {
	if ( !Passed ) {
		fprintf(TraceFile, "FAIL:     %s\n", TestName);
		fprintf(TraceFile, "REASON:   %s\n", Cause);
		fprintf(TraceFile, "\n");
		fflush(TraceFile);
	}
}

void CheckResult(
	Boolean 	Passed,
	char*		TestName,
	char*		Cause,
	unsigned	Expected,
	unsigned	Actual
) {
	if ( !Passed ) {
		fprintf(TraceFile, "FAIL:     %s\n", TestName);
		fprintf(TraceFile, "REASON:   %s\n", Cause);
		fprintf(TraceFile, "EXPECTED: %u\n", Expected);
		fprintf(TraceFile, "ACTUAL:   %u\n", Actual);
		fprintf(TraceFile, "\n");
		fflush(TraceFile);
	}
}

Boolean SweepTest(
	char*		TestName,
	SegDesc		Before,
	SegDesc		After,
	AllocSeg	Seg
) {
	BlockPtr	Block;
	char		Buffer[100];
	unsigned	DescPos;
	unsigned	Flags;
	unsigned	FlagPos;
	unsigned	FlagShift;
	FreePtr		Free;
	FreePtr		FreeSearch;
	unsigned	FreeLength;
	unsigned	Offset;

	/* Set up Seg as described in Before */

	/* Clear the segment flag bits */
	for ( Offset = 1; Offset < ArrayLength( Seg->Flags ); Offset++ ) {
		Seg->Flags[Offset] = 0;
	}
	/* Set up blocks in Seg, and set flag bits */
	for ( DescPos = 0; Before[DescPos].Length != 0; DescPos++ ) {
		Block = (BlockPtr)&Seg->Well[ Before[DescPos].Offset / sizeof( Seg->Well[0] ) ];
		if ( Before[DescPos].Length < 256 ) {
			Block->Size = Log2[ Before[DescPos].Length ];
		} else {
			Block->Size = Log2[ Before[DescPos].Length >> 8 ] + 8;
		}
		if ( Before[DescPos].Kind != FreeBlock ) {
			ASegAllocSet( Seg, Block );
		}
		if ( Before[DescPos].Kind == RefBlock ) {
			ASegMarkSet( Seg, Block );
		}
	}

	ASegSweep( Seg );
		/* Sweep it up, so we can check on you */

	/*
	The white glove test.  Make sure that Seg is homomorphic to
	After.  That is, make sure their is a 1 to 1 correspondence,
	between allocated blocks in Seg and After.  Also verify that all
	free blocks in After appear the appropriate free list in Seg.

	This check is not complete, in that it fails to check that all
	free blocks in Seg appear in After.
	*/

	DescPos = 0;
	FlagPos = 0;
	FlagShift = 0;
	Flags = Seg->Flags[0];
	Offset = 0;
	while ( True ) {
		/* Get to the next allocated block in Seg */
		while ( ! ( Flags & UnitAlloc ) ) {
			Offset += UnitSize;
			if (
				FlagShift < sizeof(Seg->Flags[0])*BitsPerByte - FlagsPerUnit
			) {
				FlagShift += FlagsPerUnit;
				Flags >>= FlagsPerUnit;
			} else {
				FlagShift = 0;
				FlagPos++;
				if ( ArrayLength(Seg->Flags) <= FlagPos ) break;
				Flags = Seg->Flags[FlagPos];
			}
		}

		/*
		Check that subsequent free blocks in After are in the
		correct free list.
		*/
		while ( After[DescPos].Kind == FreeBlock ) {
			Free = (FreePtr)( After[DescPos].Offset );
				/* Set Free to designated offset */
			/* Check that the sizes match */
			FreeLength = After[DescPos].Length;
			sprintf( Buffer, "Length of free block at %p is wrong", Free );
			CheckResult(
				FreeLength == 1 << Free->Size, TestName, Buffer,
				FreeLength, 1 << Free->Size
			);
			/* Search the free list of FreeLength for Free */
			for (
				FreeSearch = Seg->FirstFreeOfSize[ Free->Size ];
				FreeSearch != NULL;
				FreeSearch = FreeSearch->Next
			) {
				if ( FreeSearch == Free) break;
			}
			sprintf(
				Buffer,
				"Can't find free block %Fp in free list %u of segment %Fp",
				Free, FreeLength, Seg
			);
			CheckCondition( FreeSearch != NULL, TestName, Buffer );

			DescPos++;
		};

	if ( Seg->Flags <= Offset || After[DescPos].Kind == EndBlock ) break;
		/*
		Exit when there are no more blocks in Seg or After
		*/

		/*
		Check that the block in Seg and After have the same offset.
		*/
		CheckResult(
			After[DescPos].Offset == Offset,
			TestName, "Block starts at wrong offset",
			After[DescPos].Offset, Offset
		);

		/* Check that the blocks length is correct */
		Block = (BlockPtr)( Offset );
		sprintf( Buffer, "Length of allocated block at %p is wrong", Block );
		CheckResult(
			After[DescPos].Length == ( 1 << Block->Size ),
			TestName, Buffer, After[DescPos].Length, 1 << Block->Size
		);

		DescPos++;
		Offset += UnitSize;
		if (
			FlagShift < sizeof(Seg->Flags[0])*BitsPerByte - FlagsPerUnit
		) {
			FlagShift += FlagsPerUnit;
			Flags >>= FlagsPerUnit;
		} else {
			FlagShift = 0;
			FlagPos++;
			if ( ArrayLength(Seg->Flags) <= FlagPos ) break;
			Flags = Seg->Flags[FlagPos];
		}
	}

	Offset = After[DescPos].Offset;
	Block = (BlockPtr)( Offset );
	sprintf( Buffer, "Block %p is missing", Block );
	CheckCondition( After[DescPos].Kind == EndBlock, TestName, Buffer );

	return True;
}
			

SegDesc AllFree = {
	{FreeBlock,                     0, (unsigned short)32768L},
	{FreeBlock, (unsigned short)32768L,                 16384},
	{FreeBlock, (unsigned short)32768L+16384,           8192},
	{FreeBlock, (unsigned short)32768L+16384+8192,      4096},
	{FreeBlock, (unsigned short)32768L+16384+8192+4096, 2048},
	{EndBlock, 0, 0}
};

SegDesc AllUnref = {
	{UnrefBlock,                     0, (unsigned short)32768L},
	{UnrefBlock, (unsigned short)32768L,                 16384},
	{UnrefBlock, (unsigned short)32768L+16384,           8192},
	{UnrefBlock, (unsigned short)32768L+16384+8192,      4096},
	{UnrefBlock, (unsigned short)32768L+16384+8192+4096, 2048},
	{EndBlock, 0, 0}
};

SegDesc AllRef1 = {
	{RefBlock,                     0, (unsigned short)32768L},
	{RefBlock, (unsigned short)32768L,                 16384},
	{RefBlock, (unsigned short)32768L+16384,           8192},
	{RefBlock, (unsigned short)32768L+16384+8192,      4096},
	{RefBlock, (unsigned short)32768L+16384+8192+4096, 2048},
	{EndBlock, 0, 0}
};

SegDesc AllRef2 = {
	{RefBlock,                     0,  (unsigned short)16384},
	{RefBlock, (unsigned short)16384,  (unsigned short)16384},
	{RefBlock, (unsigned short)32768L,                 16384},
	{RefBlock, (unsigned short)32768L+16384,           4096},
	{RefBlock, (unsigned short)32768L+16384+4096,      4096},
	{RefBlock, (unsigned short)32768L+16384+8192,      4096},
	{RefBlock, (unsigned short)32768L+16384+8192+4096, 2048},
	{EndBlock, 0, 0}
};

SegDesc Mixed1Before = {
	{UnrefBlock,                     0, 0x40},
	{RefBlock,                    0x40, 0x10},
	{RefBlock,                    0x50, 0x08},

	{UnrefBlock,                  0x58, 0x08},
	{UnrefBlock,                  0x60, 0x10},
	{UnrefBlock,                  0x70, 0x10},
	{UnrefBlock,                  0x80, 0x10},
	{UnrefBlock,                  0x90, 0x10},
	{UnrefBlock,                  0xA0, 0x20},
	{UnrefBlock,                  0xC0, 0x40},
	{UnrefBlock,                 0x100, 0x20},
	{UnrefBlock,                 0x120, 0x08},

	{RefBlock,                   0x128, 0x08},
	{UnrefBlock,                 0x130, 0x10},
	{FreeBlock,                  0x140, 0x20},
	{RefBlock,                   0x160, 0x20},
	{UnrefBlock,                 0x180, 0x80},
	{RefBlock,                   0x200, 0x200},
	{FreeBlock,                  0x400, 0x400},
	{FreeBlock,                  0x800, 0x800},
	{FreeBlock,                 0x1000, 0x1000},
	{FreeBlock,                 0x2000, 0x2000},
	{FreeBlock,                 0x4000, 0x4000},
	{FreeBlock,                 0x8000, 0x4000},
	{FreeBlock,                 0xC000, 0x2000},
	{FreeBlock,                 0xE000, 0x1000},
	{FreeBlock,                 0xF000, 0x0800},
	{EndBlock, 0, 0}
};

/* 100=256  400=1024  1000=4096  4000=16384  8000=23768  10000=65536 */

SegDesc Mixed1After = {
	{FreeBlock,                      0, 0x40},
	{RefBlock,                    0x40, 0x10},
	{RefBlock,                    0x50, 0x08},
	{FreeBlock,                   0x58, 0x08},
	{FreeBlock,                   0x60, 0x20},
	{FreeBlock,                   0x80, 0x80},
	{FreeBlock,                  0x100, 0x20},
	{FreeBlock,                  0x120, 0x08},
	{RefBlock,                   0x128, 0x08},
	{FreeBlock,                  0x130, 0x10},
	{FreeBlock,                  0x140, 0x20},
	{RefBlock,                   0x160, 0x20},
	{FreeBlock,                  0x180, 0x80},
	{RefBlock,                   0x200, 0x200},
	{FreeBlock,                  0x400, 0x400},
	{FreeBlock,                  0x800, 0x800},
	{FreeBlock,                 0x1000, 0x1000},
	{FreeBlock,                 0x2000, 0x2000},
	{FreeBlock,                 0x4000, 0x4000},
	{FreeBlock,                 0x8000, 0x4000},
	{FreeBlock,                 0xC000, 0x2000},
	{FreeBlock,                 0xE000, 0x1000},
	{FreeBlock,                 0xF000, 0x0800},
	{EndBlock, 0, 0}
};

int main(int argc, char** argv)
{
	Boolean		Result;
	AllocSeg	Seg;

	TraceFile = fopen("trace.txt", "w");

	Seg = 0;
	ASegInitSeg(Seg);

	Result = SweepTest( "Sweep a mixed segment", Mixed1Before, Mixed1After, Seg );
	fprintf(TraceFile, "Swept mixed segment\n");
	ASegDumpSeg( Seg, TraceFile, 0 );
	fprintf(TraceFile, "\n");

	Result = SweepTest( "Sweep a free segment", AllFree, AllFree, Seg );
	fprintf(TraceFile, "Swept free segment\n");
	ASegDumpSeg( Seg, TraceFile, 0 );
	fprintf(TraceFile, "\n");

	Result = SweepTest( "Sweep an unref'd segment", AllUnref, AllFree, Seg );
	fprintf(TraceFile, "Swept unref'd segment\n");
	ASegDumpSeg( Seg, TraceFile, 0 );
	fprintf(TraceFile, "\n");

	Result = SweepTest( "Sweep a full segment - 1", AllRef1, AllRef1, Seg );
	fprintf(TraceFile, "Swept full segment - 1\n");
	ASegDumpSeg( Seg, TraceFile, 0 );
	fprintf(TraceFile, "\n");

	Result = SweepTest( "Sweep a full segment - 2", AllRef2, AllRef2, Seg );
	fprintf(TraceFile, "Swept full segment - 2\n");
	ASegDumpSeg( Seg, TraceFile, 0 );
	fprintf(TraceFile, "\n");

	return Result;
}
