/***************************************************************************

 FILENAME:		hwx_bw.c
 DESCRIPTION:   Browser window program source module for DDJ HWX example
				program.

****************************************************************************
	Copyright 1992 XVT Software Inc. All rights reserved.  XVT is a
	trademark of XVT Software Inc.

			XVT Software Inc., Box 18750, Boulder CO 80308 USA
					(303) 443-4223  fax: (303) 443-0969
***************************************************************************/

#include "xvt.h"
#include "hwx_bw.h"
#include "hwx_bwz.h"
#include "hwx_bwi.h"
#include "hwx_ad.h"
#include "hwx_std.h"
#include "hwx_glob.h"
#include "hwx_util.h"
#include "hwx_ev.h"

#include "setjmp.h"

PRIVATE jmp_buf main_read_loop;

/**************************************************************************/
WINDOW
BW_new (WINDOW parent_win)
{
	BW_DATA *BW_data_p;

	/* Allocate a BW data object, and then attach it to the BW via the
	 * creation function.
	 */
	if ((BW_data_p = (BW_DATA*)xvt_zmalloc (sizeof (BW_DATA))) == NULL)
		xvt_error ("Cannot allocate browser window data.");

	if (! BW_open_file (BW_data_p)) {
		xvt_free (BW_data_p);
		return;
	}
	BW_ev_register (BW_data_p);

	return (create_res_window (BW_DEF, parent_win, EM_ALL, EV_eh, (long)BW_data_p));
}

/**************************************************************************/
void
BW_init (WINDOW win, lplpList hwx_data, int cur_char)
{
	DRAW_CTOOLS draw_ctools;
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);

	BW_data_p->hwx_data = hwx_data;

	/* Set the default BW drawing attributes.
	 */
	get_normal_ctools (&draw_ctools);
	BW_data_p->draw_ctools = draw_ctools;

	/* Set the current character in the text edit field.
	 */
	BW_data_p->cur_char = cur_char;
	BW_data_p->init_flag = FALSE;
}

/**************************************************************************/
BOOLEAN
BW_open_file (BW_DATA *BW_data_p)
{
	DIRECTORY dir;
	BOOLEAN OK_flag = FALSE;
	FILE *file_p;
	FILE_SPEC fs;

	strcpy (fs.type, "dat");
	get_dir (&dir);
	fs.dir = dir;

	switch (open_file_dlg (&fs, "Select handwriting data file...")) {
	case FL_BAD:
		xvt_error ("Error getting file name.");
		OK_flag = FALSE;
		break;

	case FL_CANCEL:
		OK_flag = FALSE;
		break;

	case FL_OK:
		OK_flag = TRUE;
		break;
	}
	if (OK_flag == TRUE) {
		chg_dir (&(fs.dir));
		if ((file_p = fopen (fs.name, "rb")) == NULL) {
			xvt_error ("Can't open handwriting file \"%s\".", fs.name);
			OK_flag = FALSE;
		}
		else {
			BW_data_p->file_p = file_p;
			BW_data_p->fs = fs;
		}
	}
	return (OK_flag);
}

/**************************************************************************/
PRIVATE void
BW_ev_register (BW_DATA *BW_data_p)
{
	EV *ev;
	int count = 0;

	ev = EV_create (BW_SIZE_EV);
	BW_data_p->ev = ev;

	EV_set_eh (ev, count++, E_CREATE,  0, BW_create_eh);
	EV_set_eh (ev, count++, E_FOCUS,   0, BW_focus_eh);
	EV_set_eh (ev, count++, E_UPDATE,  0, BW_update_eh);
	EV_set_eh (ev, count++, E_DESTROY, 0, BW_destroy_eh);

	EV_set_eh (ev, count++, E_COMMAND, M_FILE_OPEN,  BW_propagate_eh);
	EV_set_eh (ev, count++, E_COMMAND, M_FILE_QUIT,  BW_propagate_eh);

	EV_set_eh (ev, count++, E_COMMAND, M_FILE_PRINT, BW_command_file_print_eh);
	EV_set_eh (ev, count++, E_COMMAND, M_OPT_ATTRS,  BW_command_opt_attrs_eh);

	EV_set_eh (ev, count++, E_CONTROL, BW_BUTTON_ZOOMIN,  BW_control_zoomin_eh);
	EV_set_eh (ev, count++, E_CONTROL, BW_BUTTON_ZOOMOUT, BW_control_zoomout_eh);
	EV_set_eh (ev, count++, E_CONTROL, BW_EDIT_CHAR,      BW_control_editchar_eh);
}

/**************************************************************************/
PRIVATE long
BW_create_eh (WINDOW win, EVENT *ev_p)
{
	FILE_SPEC fs;
	FILE *file_p;
	char title[SZ_TITLE], path[SZ_PATHNAME];
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);
	lplpList hwx_data;

	fs = BW_data_p->fs;
	file_p = BW_data_p->file_p;

	/* Set the window title using the data file name.
	 */
	dir_to_str (&(fs.dir), path, sizeof (path));
	sprintf (title, "Browser - %s\\%s", path, fs.name);
	set_title (win, title);

	if ((hwx_data = BW_load_data (win, file_p)) == NULL) {
		close_window (win);
		return (EV_CONSUMED);
	}

	BW_init (win, hwx_data, 'A');

	/* Enable/disable BW-specific menu items.
	 */
	win_menu_enable (win, M_FILE_OPEN, TRUE);
	win_menu_enable (win, M_FILE_PRINT, TRUE);

	/* Clear the window, set the starting values of the controls
	 * in the BW window, and create the 3 child windows for the
	 * graphics output.
	 */
	clear_window (win, (COLOR)get_value (NULL_WIN, ATTR_BACK_COLOR));
	update_window (win);

	BW_data_p->BWZ_win = BWZ_new (win, 'A', 0);
	BW_data_p->BWI_win = BWI_new (win, 'A', 0);

	BW_do_status_update (win);

	return (EV_CONSUMED);
}

/**************************************************************************/
PRIVATE long
BW_focus_eh (WINDOW win, EVENT *ev_p)
{
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);

	if ((ev_p->v.active == TRUE) &&
		(BW_data_p->init_flag == TRUE))
		set_front_window (get_ctl_window (win, BW_EDIT_CHAR));
	else
		BW_data_p->init_flag = TRUE;

	return (EV_CONSUMED);
}

/**************************************************************************/
PRIVATE long
BW_update_eh (WINDOW win, EVENT *ev_p)
{
	BW_paint (win);
	return (EV_CONSUMED);
}

/**************************************************************************/
PRIVATE long
BW_destroy_eh (WINDOW win, EVENT *ev_p)
{
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);

	/* Free the HWX data structure (if it exists), and then free the
	 * BW data object itself.
	 */
	BW_free_data (win);
	xvt_free ((char*)BW_data_p);

	return (EV_CONSUMED);
}

/**************************************************************************/
PRIVATE long
BW_propagate_eh (WINDOW win, EVENT *ev_p)
{
	return (EV_PROPAGATE);
}

/**************************************************************************/
PRIVATE long
BW_command_file_print_eh (WINDOW win, EVENT *ev_p)
{
	xvt_note ("Due to the tight deadline for the Dr. Dobb's HWX programmng project, printing was not implemented.  Of course, XVT provides a rich set of portable printing functionality.");
	return (EV_CONSUMED);
}

/**************************************************************************/
PRIVATE long
BW_command_opt_attrs_eh (WINDOW win, EVENT *ev_p)
{
	AD_new (win);
	return (EV_CONSUMED);
}

/**************************************************************************/
PRIVATE long
BW_control_zoomin_eh (WINDOW win, EVENT *ev_p)
{
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);

	enable_window (get_ctl_window (win, BW_BUTTON_ZOOMIN), FALSE);
	enable_window (get_ctl_window (win, BW_BUTTON_ZOOMOUT), TRUE);

	BWZ_do_zoom (BW_data_p->BWZ_win, BWZ_ZOOMIN_FACTOR);
	BW_do_status_update (win);

	return (EV_CONSUMED);
}

/**************************************************************************/
PRIVATE long
BW_control_zoomout_eh (WINDOW win, EVENT *ev_p)
{
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);

	enable_window (get_ctl_window (win, BW_BUTTON_ZOOMIN), TRUE);
	enable_window (get_ctl_window (win, BW_BUTTON_ZOOMOUT), FALSE);

	BWZ_do_zoom (BW_data_p->BWZ_win, BWZ_ZOOMOUT_FACTOR);
	BW_do_status_update (win);

	return (EV_CONSUMED);
}

/**************************************************************************/
PRIVATE long
BW_control_editchar_eh (WINDOW win, EVENT *ev_p)
{
	char s[2];
	WINDOW ctl_win;
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);
	CONTROL_INFO ci;

	ci = ev_p->v.ctl.ci;
	ctl_win = ci.win;

	if ((ci.v.edit.focus_change == TRUE) &&
		(ci.v.edit.active == TRUE)) {
		win_select_item_text (ctl_win, 0, 1);
		return;
	}
	if (ci.v.edit.focus_change == FALSE) {
		get_title (ctl_win, s, 2);

		if (s[0] == NULL)
			return;

		if (BW_data_p->hwx_data[s[0]] == NULL)
		{
			xvt_beep();

			s[0] = (char)BW_data_p->cur_char;
			s[1] = NULL;
			set_title (ctl_win, s);
			win_select_item_text (ctl_win, 0, 1);
			return;
		}
		BW_data_p->cur_char = s[0];
		set_title (ctl_win, s);

		/* Tell the BW, BWZ and BWI windows to "update" themselves
		 * with their data.
		 */
		BW_do_status_update (win);
		BWZ_do_update (BW_data_p->BWZ_win, s[0], NO_CHANGE);
		BWI_do_update (BW_data_p->BWI_win, s[0], NO_CHANGE, NO_OPTIMIZE);
	}
	return (EV_CONSUMED);
}

/**************************************************************************/
void
BW_paint (WINDOW win)
{
	clear_window (win, (COLOR)get_value (NULL_WIN, ATTR_BACK_COLOR));
}

/**************************************************************************/
lplpList
BW_load_data (WINDOW win, FILE *file_p)
{
	VHPoint  Pt;
	lpList   char_instance;
	lpList   stroke;
	INT16    i, j, k, num_strokes, num_pts, which;
	INT16    char_code, num_instances;
	lplpList hwx_data;
	WINDOW   dlg;

	wait_cursor();

	/* Return to this location if an EOF is reached during some deeply-
	 * nested file read.
	 */
	if (setjmp (main_read_loop) == EOF_FLAG)
	{
		if (! feof (file_p)) {
			xvt_note ("Error reading HWX data file!");
			return (NULL);
		}
		else
		{
			STD_do_status_update (dlg, STD_CLOSE_WIN, 0, 0);
			fclose (file_p);
			return (hwx_data);
		}
	}

	if ((hwx_data = xvt_zmalloc (sizeof (lpList) * MAX_CHARS)) == NULL)
		xvt_fatal ("Unrecoverable memory allocation error for HWX data array!");

	if (! BW_read_file_header (file_p)) {
		xvt_note ("Error reading HWX data file...file header problem.");
		return (NULL);
	}

	dlg = STD_new (TASK_WIN);

	while (TRUE)
	{
		char_code = BW_read_file_word (file_p);
		num_instances = BW_read_file_word (file_p);

		STD_do_status_update (dlg, STD_UPDATE, char_code, num_instances);

		which = (char_code < MAX_CHARS) ? char_code : 1;
		hwx_data[which] = xvt_zmalloc (sizeof(INT16) + (num_instances * sizeof(long)));

		if (! hwx_data[which]) {
			xvt_fatal ("Memory allocation failure #1");
		}
		hwx_data[which]->num_items = num_instances;

		for (i = 0; i < num_instances; i++)
		{
			num_strokes = BW_read_file_word (file_p);
			char_instance = xvt_zmalloc (sizeof(INT16) + (num_strokes * sizeof(long)));

			if (! char_instance) {
				xvt_fatal ("Memory allocation failure #2");
			}
			char_instance->num_items = num_strokes;

			for (j = 0; j < num_strokes; j++)
			{
				num_pts = BW_read_file_word (file_p);

				Pt.h = BW_read_file_word (file_p);
				Pt.v = BW_read_file_word (file_p);

				stroke = xvt_zmalloc (sizeof(INT16) + num_pts * sizeof(long));

				if (! stroke) {
					xvt_fatal ("Memory allocation failure #3");
				}
				stroke->num_items = num_pts;
				stroke->items[0] = *(void**)&Pt;

				for (k = 1; k < num_pts; k++)
				{
					INT16 s = BW_read_file_word (file_p);
					LPSTR ps = (LPSTR)&s;

					Pt.h += ps[1]; Pt.v += ps[0];
					stroke->items[k] = *(void**)&Pt;
				}
				char_instance->items[j] = stroke;
			}
			hwx_data[which]->items[i] = char_instance;
		}
	}
}

/**************************************************************************/
PRIVATE BOOLEAN
BW_read_file_header (FILE *file_p)
{
	INT16 signature, header_size, version_major, version_minor;

	signature = BW_read_file_word (file_p);
	if (signature != FILE_SIGNATURE)
		return (FALSE);

	header_size = BW_read_file_word (file_p); //number of bytes should be even
	if (header_size & 01)
		return (FALSE);

	header_size >>= 1;  // number of words
	header_size--;

	version_major = BW_read_file_word (file_p); header_size--;
	version_minor = BW_read_file_word (file_p); header_size--;

	while (header_size--)
		BW_read_file_word (file_p);

	/* The following line serves only to get rid of a compiler warning.
	 */
	version_major++; version_minor++;

	return (TRUE);
}

/**************************************************************************/
INT16
BW_read_file_word (FILE *file_p)
{
	char s[2];

	s[0] = s[1] = 0;

	if (fread (&s, sizeof(char), 2, file_p) != 2)
		longjmp (main_read_loop, EOF_FLAG);

	return (INT16)( ((BYTE)s[1]) | ((WORD)((BYTE)s[0]) << 8) );
}

/**************************************************************************/
void
BW_free_data (WINDOW win)
{
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);
	INT16 i,j,k;
	lplpList hwx_data;

	hwx_data = BW_data_p->hwx_data;

	for (i = 0; i < MAX_CHARS; i++)
	{
		if (hwx_data[i])
		{
			for (j = 0; j < hwx_data[i]->num_items; j++)
			{
				lpList char_instance = hwx_data[i]->items[j];
				for (k = 0; k < char_instance->num_items; k++)
					xvt_free ((char*)char_instance->items[k]);
				xvt_free ((char*)char_instance);
			}
			xvt_free ((char*)hwx_data[i]);
			hwx_data[i] = NULL;
		}
	}
	xvt_free ((char*)hwx_data);
	BW_data_p->hwx_data = (lplpList)NULL;
}

/**************************************************************************/
PUBLIC void
BW_do_status_update (WINDOW win)
{
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);
	BWI_DATA *BWI_data_p;
	char s[4];

	/* Update all of the "status controls" in the BW, and
	 * set the focus to the edit field.
	 */
	BWI_data_p = (BWI_DATA*)get_app_data (BW_data_p->BWI_win);

	set_title (get_ctl_window (win, BW_TEXT_ASCII_VAL),
			   itoa (BW_data_p->cur_char, s, 10));

	set_title (get_ctl_window (win, BW_TEXT_ID_VAL),
			   itoa (BWI_data_p->cur_instance, s, 10));

	win_select_item_text (get_ctl_window (win, BW_EDIT_CHAR), 0, 1);
	set_front_window (get_ctl_window (win, BW_EDIT_CHAR));
}

/**************************************************************************/
PUBLIC void
BW_do_attr_update (WINDOW win, DRAW_CTOOLS *draw_ctools)
{
	BW_DATA *BW_data_p = (BW_DATA*)get_app_data (win);

	/* Set the BW data to the supplied tools, and invalidate the BWI
	 * and BWZ child windows.
	 */
	BW_data_p->draw_ctools = *(draw_ctools);

	invalidate_rect (BW_data_p->BWZ_win, NULL);
	invalidate_rect (BW_data_p->BWI_win, NULL);
}