/***************************************************************************

 FILENAME:		hwx_gr.c
 DESCRIPTION:   Graphics and handwriting data access source module for
				DDJ HWX example program.  Based on the DDJ-supplied code
				for this project.

****************************************************************************
	Copyright 1992 XVT Software Inc. All rights reserved.  XVT is a
	trademark of XVT Software Inc.

			XVT Software Inc., Box 18750, Boulder CO 80308 USA
					(303) 443-4223  fax: (303) 443-0969
***************************************************************************/

#include "xvt.h"
#include "hwx_gr.h"
#include "hwx_bwz.h"
#include "hwx_bwi.h"
#include "hwx_glob.h"

#include "h_config.h"

#include <stdio.h>
#include <setjmp.h>


#include "h_stddef.h"
#include "h_list.h"
#include "h_mem.h"
#include "h_fmtstr.h"
#include "h_grafic.h"
#include "h_keys.h"

/***************Data that is local to this module. **************/

private int theCurrentChar;
private int theCurrentInstance;



/***************************************************************
|    IMPLEMENTATION OF ROUTINES FROM THE SPEC
\****************************************************************/

private void myMoveTo(HDISPLAY display_entity,INT16 h,INT16 v)
{
	/* this implementation does not use handle to display entity */
	gr_MoveTo(h,v);
}
/****************************************************************/
private void myLineTo(HDISPLAY display_entity,INT16 h,INT16 v)
{
	/* this implementation does not use handle to display entity */
	gr_LineTo(h,v);
}
/****************************************************************/
private void ShowUsage(void)
{
	printf("\nUsage: dosbrows <HWX filename>");
	printf("\n");
}
/****************************************************************/
public void RegisterCallback(
	   HDISPLAY display_entity,
	   void (*pf_move_to)(HDISPLAY,INT16,INT16),
	   void (*pf_line_to)(HDISPLAY,INT16,INT16))
{
	/*stash pointers away, ignore handle to display_entity */
	pf_MoveTo = pf_move_to;
	pf_LineTo = pf_line_to;
}
/****************************************************************/
public void LoadHWXData(LPSTR file_name)
{
	FILE *data_file;

	data_file = fopen(file_name,"rb");
	if (!data_file)
	FatalError(fmtStr("\n***File %s not found!\n",file_name));

	FreeMemStructures();
	ParseDataFormat(data_file);
	fclose(data_file);
}
/****************************************************************/
public void UnloadHWXData(void)
{
    FreeMemStructures();
}
/****************************************************************/
public int GetInstanceCount(int ch)
{
    if(CharData[ch]) return CharData[ch]->num_items;
    else return 0;
}
/****************************************************************/
public lpList GetInstanceData(int ch,int inst)
{
    lpList instance_data;
    if(!CharData[ch]) return 0;
    instance_data = CharData[ch]->items[inst];
    if(!instance_data) return 0;
    /* THe instance_data variable now points to List of strokes, which 
     * themselves point to a List of points.
     * The number of list elements is in instance_data->num_items.
     */
    return instance_data;
}
/****************************************************************/
#define X_BIGBOX_START  20
#define Y_BIGBOX_START 120

#define BIGBOX_WIDTH 140
#define BIGBOX_DEPTH 140

public void DisplayInstance(int char_code,int instance_num)
{
	INT16 hshift,vshift;
	Rect bbox_tab;  /* in world/object/tablet coordinates */
	Rect bbox_scr;  /* in screen coordinates */
	Rect bbox_out;  /* shifted for output */

	lpList char_instance;

	ShowHeading(char_code,instance_num);

	char_instance = CharData[char_code]->items[instance_num];
    
    hshift = X_BIGBOX_START;
    vshift = Y_BIGBOX_START;
    
    gr_SetTabToScrConversion(2,2);
    CalcInstanceBoundingBox(char_instance,&bbox_tab);

    bbox_scr = bbox_tab; /*structure copy */
    gr_ConvertRectTabletToScreen(&bbox_scr);
    gr_SetRect(&bbox_out,
	hshift,vshift,
	hshift+BIGBOX_WIDTH,
	vshift+BIGBOX_DEPTH);
    if(instance_num==theCurrentInstance)
    {
	Rect r = bbox_out;
	gr_InflateRect(&r,-2);	
	gr_SetColorBlue();
	gr_DrawRect(&r);
	}
    gr_SetColorWhite();
    gr_DrawRect(&bbox_out);

    hshift += ((BIGBOX_WIDTH  - (bbox_scr.right  - bbox_scr.left)) / 2);
    vshift += ((BIGBOX_DEPTH - (bbox_scr.bottom - bbox_scr.top )) / 2);    

	gr_SetRect(&bbox_out,
	hshift,vshift,
	hshift+(bbox_scr.right  - bbox_scr.left),
	vshift+(bbox_scr.bottom - bbox_scr.top));
    gr_SetColorGray();
    gr_DrawRect(&bbox_out);
    gr_SetColorYellow();
    NormalizeAndDisplayInstance(char_instance,&bbox_tab,hshift,vshift,1);
}

/***************************************************************
|    OTHER SUPPORTING ROUTINES
\****************************************************************/
private void FreeMemStructures(void)
{
	INT16 i,j,k;
	for (i = 0; i < MAX_CHARS; i++) if (CharData[i])
	{
		for (j = 0; j < CharData[i]->num_items; j++)
	{
			lpList char_instance = CharData[i]->items[j];
			for (k = 0; k < char_instance->num_items; k++)
				mem_Free(char_instance->items[k]);
			mem_Free(char_instance);
		}
		mem_Free(CharData[i]);
		CharData[i] = NULL;
	}
}
/****************************************************************/
private INT16  ReadWordFromFile(FILE *data_file)
{
    char s[2];
    s[0] = s[1] = 0;
    if(fread(&s, sizeof(char), 2, data_file) != 2)
	longjmp( main_loop, -1 );
    num_bytes += 2;
    return (INT16)( ((BYTE)s[1]) | ((WORD)((BYTE)s[0])<<8) );
}
/****************************************************************/
private void ReadHeader(FILE *data_file)
{
    INT16 signature,header_size,version_major,version_minor;
    signature = ReadWordFromFile(data_file);
    if(signature != FILE_SIGNATURE) FatalError("***Header format error!***");
    header_size = ReadWordFromFile(data_file); //number of bytes should be even
    if(header_size & 01) FatalError("***Header format error #2!***");
    header_size >>= 1;  // number of words
    header_size--;
    version_major = ReadWordFromFile(data_file); header_size--;
    version_minor = ReadWordFromFile(data_file); header_size--;    
    while(header_size--) ReadWordFromFile(data_file);
    //the following line serves only to get rid of a compiler warning
    version_major++; version_minor++; 
}
/****************************************************************/
private void ParseDataFormat(FILE *data_file)
{
	VHPoint Pt;
    lpList   char_instance;
    lpList   stroke;
    INT16    i,j,k,num_strokes,num_pts,which;
    INT16    char_code,num_instances;

    if( setjmp( main_loop )==0 ) 
    {
	ReadHeader(data_file);
	while (1)
        {
        char_code = ReadWordFromFile(data_file);
        num_instances = ReadWordFromFile(data_file);

        which = (char_code < MAX_CHARS) ? char_code : 1;
        CharData[which] = mem_Alloc(sizeof(INT16) 
			    + (num_instances * sizeof(long)));
	if(!CharData[which]) 
			    mem_AllocFailed(sizeof(INT16) 
			    + (num_instances * sizeof(long)));
        CharData[which]->num_items = num_instances;
        for (i = 0; i < num_instances; i++) 
	{
            num_strokes = ReadWordFromFile(data_file);

            char_instance = mem_Alloc(sizeof(INT16) + 
				(num_strokes * sizeof(long)));
	    if(!char_instance) 
			    mem_AllocFailed(sizeof(INT16) 
			    + (num_strokes * sizeof(long)));

			char_instance->num_items = num_strokes;

            for (j = 0; j < num_strokes; j++) 
	    {
                num_pts = ReadWordFromFile(data_file);
		Pt.h = ReadWordFromFile(data_file);
		Pt.v = ReadWordFromFile(data_file);

                stroke = mem_Alloc(sizeof(INT16) + num_pts * sizeof(long));
		if(!stroke) 
		    mem_AllocFailed(sizeof(INT16) + num_pts * sizeof(long));
                stroke->num_items = num_pts;
                stroke->items[0] = *(void**)&Pt;
                for (k = 1; k < num_pts; k++) 
		{
		    INT16 s = ReadWordFromFile(data_file);
		    LPSTR ps = (LPSTR )&s;

					Pt.h += ps[1]; Pt.v += ps[0];
                    stroke->items[k] = *(void**)&Pt;
                } /* for loop (k) */
                char_instance->items[j] = stroke;

            } /* for loop (j) */
            CharData[which]->items[i] = char_instance;
           }  /* for loop (i) */                  
         } /*while loop */
       } /*if*/
       else if (!feof(data_file))
       {
	   FatalError(fmtStr("\n***Err reading file at loc %ld!",num_bytes));
	   }
}
/****************************************************************/
private void FatalError(char* msg)
{
    if(graphics_enabled) gr_Close();

    fprintf(stderr,"\n***Fatal Error: ");
    fprintf(stderr,msg);
    exit(0);
}
/****************************************************************/

#define MIN_VALUE (-0x7FFF)
#define MAX_VALUE ( 0x7FFF)

private void CalcInstanceBoundingBox(lpList char_instance,lpRect bb)
{
    INT16 i,j;
    INT16 x_min,x_max,y_min,y_max;
    INT16 height, width;

     x_min = y_min = MAX_VALUE;
     x_max = y_max = MIN_VALUE;

    /*------for a given char, find the maximum and minimum -------*/
    for (i = 0; i < char_instance->num_items; i++) 
    {
        lpList    stroke =   char_instance->items[i];

        for (j = 0; j < stroke->num_items; j++) 
	{
            VHPoint p = *(lpVHPoint)&stroke->items[j];
            if (p.h > x_max) x_max = p.h;
            if (p.h < x_min) x_min = p.h;
            if (p.v > y_max) y_max = p.v;
            if (p.v < y_min) y_min = p.v;
        }
    }
    bb->left  = x_min;    bb->top   = y_min; 
    bb->right = x_max;    bb->bottom = y_max;
}
/****************************************************************/
private void NormalizeAnInstance(lpList char_instance,lpRect bb)
{
    /*-------for a given char instance, normalize to xmin and ymin----*/
    INT16 i,j;
    INT16 x_min,y_min;

    x_min = bb->left;
    y_min = bb->top;

    for (i = 0; i < char_instance->num_items; i++) 
    {
        lpList    stroke =   char_instance->items[i];
	
        for (j = 0; j < stroke->num_items; j++) 
	{
            VHPoint p = *(lpVHPoint)&stroke->items[j];
	    /* subtract minimum from the character */
	    p.h -= (x_min - 100);
	    p.v -= (y_min - 100);	    

            *(lpVHPoint)&stroke->items[j] = p;
        }
    }
}
/****************************************************************/
private void NormalizeAndDisplayInstance(lpList char_instance, lpRect bbox,
		    INT16 hshift,INT16 vshift,bool use_callbacks)
{
    INT16 i,j;
    INT16 x_min,y_min;
    
    x_min = bbox->left;
    y_min = bbox->top;
    
    for (i = 0; i < char_instance->num_items; i++) 
    {
        lpList    stroke =   char_instance->items[i];
	
        for (j = 0; j < stroke->num_items; j++) 
	{
            VHPoint p = *(lpVHPoint)&stroke->items[j];
	    p.h -= x_min;
	    p.v -= y_min;
	    gr_CvtTabPtToScreen(&p);
	    gr_ShiftPoint(&p,hshift,vshift);	

	    gr_SetColorYellow();

	    if(use_callbacks)
	    {
		if(j==0) (*pf_MoveTo)(0,p.h,p.v);
		else     (*pf_LineTo)(0,p.h,p.v);
	    }
	    gr_PutPoint(&p);	
        }
    }
}
    
/***************************************************************/
#define H_TABLE_START 180
#define V_TABLE_START  60
#define NUM_INST_PER_ROW 5
#define CELL_WIDTH   80
#define CELL_HEIGHT  80

private void ShowInstanceInTable(int instance_num,lpList char_instance)
{
    INT16 hshift,vshift;
    Rect bbox_tab;  /* in world/object/tablet coordinates */
    Rect bbox_scr;  /* in screen coordinates */
    Rect bbox_out;  /* shifted for output */

    hshift = (instance_num % NUM_INST_PER_ROW) * CELL_WIDTH;
    vshift = (instance_num / NUM_INST_PER_ROW) * CELL_HEIGHT;
    hshift += H_TABLE_START;
    vshift += V_TABLE_START;

    CalcInstanceBoundingBox(char_instance,&bbox_tab);
    bbox_scr = bbox_tab; /*structure copy */
    gr_SetTabToScrConversion(3,3);
    gr_ConvertRectTabletToScreen(&bbox_scr);
    gr_SetRect(&bbox_out,
	hshift,vshift,
	hshift+CELL_WIDTH, 
	vshift+CELL_HEIGHT);

    if(instance_num==theCurrentInstance)
    {
	Rect r = bbox_out;
	gr_InflateRect(&r,-2);
	gr_SetColorBlue();
	gr_DrawRect(&r);
    }
    gr_SetColorWhite();	
    gr_DrawRect(&bbox_out);

    hshift += ((CELL_WIDTH  - (bbox_scr.right  - bbox_scr.left)) / 2);
    vshift += ((CELL_HEIGHT - (bbox_scr.bottom - bbox_scr.top )) / 2);    

    gr_SetRect(&bbox_out,
	hshift,vshift,
	hshift+(bbox_scr.right  - bbox_scr.left),
	vshift+(bbox_scr.bottom - bbox_scr.top));
    gr_SetColorGray();
    gr_DrawRect(&bbox_out);
    gr_SetColorYellow();
    NormalizeAndDisplayInstance(char_instance,&bbox_tab,hshift,vshift,0);
}

/****************************************************************/

#define HEADING_X_LOC 160
#define HEADING_Y_LOC  30

private void ShowHeading(int char_code,int instance_num)
{
    /* first erase any old stuff on screen */
    gr_SetColorBlack();
    gr_StringAt(HEADING_X_LOC, HEADING_Y_LOC ,"                          ");

    /* then, write the new stuff */
    gr_SetColorYellow();
    gr_StringAt(HEADING_X_LOC, HEADING_Y_LOC ,
       fmtStr("Char #%d    Instance #%d",char_code,instance_num));
}

/****************************************************************/
private bool PickSetOfInstances(void)
{
    INT16 ch; 

    ch = getch();
    if(ch==ESC_KEY) return FALSE;

    if(ch==0)
    {
	ch = getch();
	switch(ch)
	{
	    case LEFT_ARROW_KEY:
		theCurrentInstance = MAX(theCurrentInstance-1, 0);
		break;

	    case RIGHT_ARROW_KEY:
		theCurrentInstance 
			= MIN(theCurrentInstance+1,
			CharData[theCurrentChar]->num_items- 1);
		break;

	    case DOWN_ARROW_KEY:	
		if(theCurrentInstance < 15)
		    theCurrentInstance += NUM_INST_PER_ROW;
		break;
	    case UP_ARROW_KEY:
		if(theCurrentInstance >= NUM_INST_PER_ROW)
		    theCurrentInstance -= NUM_INST_PER_ROW;
		break;
        }	    	
    }
    else if( (0 < ch && ch < MAX_CHARS) && CharData[ch])
    {
	   theCurrentChar = ch;
	   theCurrentInstance = 0;
    }
    gr_ClearViewport();
    ShowSetOfInstances(theCurrentChar);
    DisplayInstance(theCurrentChar,theCurrentInstance);
    return TRUE;
}


/****************************************************************/
private void ShowSetOfInstances(INT16 ch)
{
    INT16 i,j;
   
    for (j = 0; j < CharData[ch]->num_items; j++)
    {
	ShowInstanceInTable(j, CharData[ch]->items[j]);
    }
}

