//-------------------------------------------------------------//
//              Autumn Hill Software Incorporated              //
//          Copyright (c) 1991 -- All Rights Reserved          //
//-------------------------------------------------------------//
// File:   Hwx.Cpp                                             //
// Desc:   Interface layer for DDJ HWX files                   //
// Date:   July, 1992                                          //
//-------------------------------------------------------------//

#include "hwx.hpp"

//======================== file I/O ===========================//

int rdword( FILE *f )
{
   unsigned char b[2];
   fread( b, 2, 1, f );
   return feof(f) ?
      -1 : (int) ( ((unsigned)(b[0]<<8)) | ((unsigned)(b[1])) );
}

int rdblock( FILE *f, char *blk, int nbytes )
{
   fread( blk, nbytes, 1, f );
   return feof(f) ? -1 : 0;
}

//================= struct HwxCharAccessEntry =================//

HwxCharAccessEntry::HwxCharAccessEntry( )
{
   ascii_code = instance_cnt = 0;
   instance_ofs = 0;
}

HwxCharAccessEntry::HwxCharAccessEntry( int ac, int ic )
{
   ascii_code = ac;
   instance_cnt = ic;
   instance_ofs = new unsigned long [ic];
   if( instance_ofs )
      for( int i=0; i<ic; i++ )
         instance_ofs[i] = 0;
}

HwxCharAccessEntry::~HwxCharAccessEntry( )
{
   if( instance_ofs )
      delete [] instance_ofs;
}

//================= struct HwxCharAccessTable =================//

HwxCharAccessTable::HwxCharAccessTable( )
{
   name   = 0;
   fptr   = 0;
   cae    = 0;
   nchars = 0;
   status = atNOINIT;
}

HwxCharAccessTable::HwxCharAccessTable( char *hwxpath, int maxchars )
{
   // status assumption
   status = atOPEN;

   // file name
   int i = strlen( hwxpath );
   name = new char [i+1];
   if( name )
      strcpy( name, hwxpath );
   else
      status = atNOMEMORY;

   // character access entry array
   nchars = 0;
   cae = new HwxCharAccessEntry [maxchars];
   if( cae )
      nchars = maxchars;
   else
      status = atNOMEMORY;

   // if all is well, open file and scan
   if( status == atOPEN )
   {
      fptr = fopen( name, "rb" );
      if( fptr == 0 )
         status = atNOTFOUND;
   }
   else
      fptr = 0;
   if( fptr )
      scan_file();
}

HwxCharAccessTable::~HwxCharAccessTable( )
{
   if( name ) delete name;
   if( cae )  delete [] cae;
   if( fptr ) { fclose( fptr );  fptr = 0; }
}

int HwxCharAccessTable::locate( int asciicode, int instance )
{
   if( asciicode > nchars ) return 0;
   if( instance > cae[asciicode].instance_cnt ) return 0;
   unsigned long ofs = cae[asciicode].instance_ofs[instance];
   return fseek(fptr,ofs,SEEK_SET) ? 0 : 1;
}

void HwxCharAccessTable::scan_file( void )
{
   // note: this function assumes successful constructor call,
   // and file is open at top-of-file...

   // file header
   int i = rdword( fptr );
   if( i != HwxSignature )
   {
      status = atBADSIGNATURE;
      return;
   }
   i = rdword( fptr );
   i -= 2;
   fseek( fptr, i, SEEK_CUR );

   // file body
   while( 1 )
   {
      // ascii code and instance count
      int ci = rdword( fptr );
      if( (ci < 0) || (ci >= nchars) ) break;
      int ni = rdword( fptr );
      if( feof(fptr) ) break;

      // initialize cae entry
      cae[ci].ascii_code = ci;
      cae[ci].instance_cnt = ni;
      cae[ci].instance_ofs = new unsigned long [ni];
      if( cae[ci].instance_ofs == 0 )
      {
         status = atNOMEMORY;
         break;
      }

      // read each instance
      int icur = 0;
      while( ni-- )
      {
         // save offset to this instance
         cae[ci].instance_ofs[icur++] = ftell(fptr);

         // stroke count
         int ns = rdword( fptr );

         // read each stroke
         while( ns-- )
         {
            // point count
            int np = rdword( fptr );

            // convert count to bytes
            np <<= 1;
            if( np > 0 ) np+=2;

            fseek( fptr, np, SEEK_CUR );
         }
      }
   }
}

//===================== struct HwxStroke ======================//

HwxStroke::HwxStroke( )
{
   xo = yo = 0;
   nvect = 0;
   vect = 0;
}

HwxStroke::HwxStroke( int nv )
{
   xo = yo = 0;
   vect = new char [nv<<1];
   nvect = vect ? nv : 0;
}

HwxStroke::~HwxStroke( )
{
   if( vect ) { delete vect; vect = 0; }
}

int HwxStroke::read( FILE *f )
{
   // wipe any existing data
   nvect = 0;
   if( vect )
   {
      delete vect;
      vect = 0;
   }

   // get point count from file
   int np = rdword(f);
   if( np <= 0 ) return 0;

   // read first point, the stroke origin
   xo = rdword( f );
   yo = rdword( f );

   // point count of 1 implies no vectors-- not likely but
   // handle it as a single vector of zero magnitude
   if( np == 1 )
   {
      vect = new char[2];
      if( vect )
      {
         nvect = 1;
         vect[0] = vect[1] = 0;
      }
      else
         nvect = 0;
   }
   else
   {
      nvect = np - 1;
      vect = new char[nvect<<1];
      if( vect )
         rdblock( f, vect, nvect<<1 );
      else
         nvect = 0;
   }
   return feof(f) ? -1 : nvect;
}

//===================== struct HwxChar ========================//

HwxChar::HwxChar( )
{
   ascii_code = 0;
   nstrokes = 0;
   strokes = 0;
}

HwxChar::HwxChar( int ac, int ns )
{
   ascii_code = ac;
   strokes = new HwxStroke [ns];
   nstrokes = strokes ? ns : 0;
}

HwxChar::~HwxChar( )
{
   if( strokes ) { delete [] strokes;  strokes = 0; }
}

// this function assumes file positioned to desired instance
int HwxChar::read( FILE *f )
{
   // wipe existing data
   nstrokes = 0;
   if( strokes )
   {
      delete [] strokes;
      strokes = 0;
   }

   // read stroke count
   int ns = rdword( f );
   if( ns <= 0 ) return 0;

   // allocate stroke array
   strokes = new HwxStroke [ns];
   nstrokes = strokes ? ns : 0;

   // read each stroke
   for( int i=0; i<nstrokes; i++ )
      strokes[i].read( f );

   return feof(f) ? -1 : nstrokes;
}

// compute a character's coordinate extent
void HwxChar::extent( int *xmin, int *ymin, int *xmax, int *ymax )
{
   int x1 = strokes[0].xo;
   int x2 = strokes[0].xo;
   int y1 = strokes[0].yo;
   int y2 = strokes[0].yo;

   for( int i=0; i<nstrokes; i++ )
   {
      int x = strokes[i].xo;
      int y = strokes[i].yo;

      if( x < x1 ) x1 = x;
      if( x > x2 ) x2 = x;
      if( y < y1 ) y1 = y;
      if( y > y2 ) y2 = y;

      int n = 0;
      for( int j=0; j<strokes[i].nvect; j++ )
      {
         x += strokes[i].vect[n++];
         y += strokes[i].vect[n++];

         if( x < x1 ) x1 = x;
         if( x > x2 ) x2 = x;
         if( y < y1 ) y1 = y;
         if( y > y2 ) y2 = y;
      }
   }

   *xmin = x1;
   *ymin = y1;
   *xmax = x2;
   *ymax = y2;
}

//=================== struct HwxCharSet =======================//

HwxCharSet::HwxCharSet( )
{
   ascii_code = instance_cnt = 0;
   charset = 0;
}

HwxCharSet::HwxCharSet( int which_ch, HwxCharAccessTable& catbl )
{
   ascii_code = which_ch;
   int ni = catbl.cae[which_ch].instance_cnt;
   charset = new HwxChar[ni];
   instance_cnt = charset ? ni : 0;
   for( int i=0; i<instance_cnt; i++ )
   {
      charset[i].ascii_code = which_ch;
      fseek( catbl.fptr,
             catbl.cae[which_ch].instance_ofs[i],
             SEEK_SET );
      charset[i].read( catbl.fptr );
   }
}

HwxCharSet::~HwxCharSet( )
{
   if( charset ) { delete [] charset;  charset = 0; }
}

//=================== struct HwxAlphabet ======================//

HwxAlphabet::HwxAlphabet( )
{
   nchars = instance = 0;
   charset = 0;
}

HwxAlphabet::HwxAlphabet( int which_inst, HwxCharAccessTable& catbl )
{
   instance = which_inst;
   int nc = catbl.nchars;
   charset = new HwxChar [nc];
   nchars = charset ? nc : 0;
   for( int i=0; i<nchars; i++ )
   {
      charset[i].ascii_code = catbl.cae[i].ascii_code;
      fseek( catbl.fptr,
             catbl.cae[i].instance_ofs[which_inst],
             SEEK_SET );
      charset[i].read( catbl.fptr );
   }
}

HwxAlphabet::~HwxAlphabet( )
{
   if( charset ) { delete [] charset; charset = 0; }
}
