//-------------------------------------------------------------//
//              Autumn Hill Software Incorporated              //
//          Copyright (c) 1991 -- All Rights Reserved          //
//-------------------------------------------------------------//
// File:   Hwx.Hpp                                             //
// Desc:   Interface layer for DDJ HWX files                   //
// Date:   July, 1992                                          //
//-------------------------------------------------------------//

#ifndef _HWX_HPP_
#define _HWX_HPP_

#include "stdio.h"    // required for stream definitions
#include "string.h"   // used by constructors

// Notes...
//
// (1) Array members of all classes are indexed either by
//     instance number starting from zero, or by ascii code
//     starting from zero.
//
// (2) Although a given object is located by ascii code index,
//     the code is still stored in the object, so that the object
//     can provide a "who am i?" capability.
//
// (3) Note that a slot exists for ascii_code=0, although there
//     will not be a definition for this code.  This is an
//     implementation convenience.

//-------------------------------------------------------------//

// hwx file access character offset table structures

struct HwxCharAccessEntry
{
   int            ascii_code;     // character code
   int            instance_cnt;   // instance count
   unsigned long *instance_ofs;   // file offset of each instance

   HwxCharAccessEntry( );
   HwxCharAccessEntry( int ac, int ic );
   ~HwxCharAccessEntry( );
};

struct HwxCharAccessTable
{
   char                *name;      // file name
   FILE                *fptr;      // stream pointer
   HwxCharAccessEntry  *cae;       // entry for each code
   int                  nchars;    // char code count
   int                  status;    // status code

   HwxCharAccessTable( );
   HwxCharAccessTable( char *hwxpath, int maxchars );
  ~HwxCharAccessTable( );
   int locate( int asciicode, int instance );

   private:
      void scan_file( void );
};

// Access Table status code values
enum AccessTableStatusCodes
{
     atNOINIT,         // nothing done
     atNOMEMORY,       // no memory, constructor failure
     atOPEN,           // file is open, all is well
     atCLOSED,         // file is closed, table not usable
     atNOTFOUND,       // file not found, table not usable
     atBADSIGNATURE,   // bad signature, table not usable
     atBADFORMAT,      // read error, table not usable
};

// Used to validate file
const int HwxSignature = 0x2121;

//-------------------------------------------------------------//

// a single stroke - array of offsets from (xo,yo)

struct HwxStroke
{
   int    xo;           // stroke x origin
   int    yo;           // stroke y origin
   int    nvect;        // number of vectors
   char  *vect;         // vector list as signed x-y offsets

   HwxStroke( );
   HwxStroke( int nv );
   ~HwxStroke( );
   int read( FILE * f );
};

// a single character - array of strokes

struct HwxChar
{
   int        ascii_code;   // character code
   int        nstrokes;     // stroke count
   HwxStroke *strokes;      // stroke array

   HwxChar( );
   HwxChar( int ac, int ns );
   ~HwxChar( );
   int read( FILE * f );
   void extent( int *xmin, int *ymin, int *xmax, int *ymax );
};

// all instances of a given character as array of characters

struct HwxCharSet
{
   int      ascii_code;        // character code
   int      instance_cnt;      // number of instances
   HwxChar *charset;           // character array

   HwxCharSet( );
   HwxCharSet( int which_ch, HwxCharAccessTable& catbl );
  ~HwxCharSet( );
};

// all characters of a given instance as array of characters

struct HwxAlphabet
{
   int         instance;    // current instance
   int         nchars;      // character count
   HwxChar    *charset;     // character array

   HwxAlphabet( );
   HwxAlphabet( int which_inst, HwxCharAccessTable& catbl );
  ~HwxAlphabet( );
};

//-------------------------------------------------------------//

// file I/O

int rdword( FILE *f );
int rdblock( FILE *f, char *blk, int nbytes );

#endif
