// ---------- gui.cpp

// ============================
// Generic User Interface (GUI)
// ============================

#ifdef ZTC
#include <iostream.hpp>
#include <iomanip.hpp>
#else
#include <iostream.h>
#include <iomanip.h>
#endif
#include <stdarg.h>
#include <ctype.h>
#include <conio.h>
#include <bios.h>
#include "gui.h"
#include "strings.h"

// ------- the GUI object
GUI gui;

// ---------------------------------
// menu user interface
// ---------------------------------
void GUI::UserMenu(char *title, ...)
{
	MenuSelection sels[MAXCOMMANDS];
	va_list ap;
	va_start(ap, title);
	int cmdno = 0;

	// ----- gather the menu from the caller's parameters
	while ((sels[cmdno].label=va_arg(ap, char *)) != NULL) {
		if (cmdno == MAXCOMMANDS)
			break;
		void *fn = va_arg(ap, void *);
		sels[cmdno++].menufunction = (void(*)()) fn;
	}

	for (;;)	{
		ClearScreen();
		SetCursor(menux, menuy);
		// ------ display the menu
		cout << title;
		for (int i = 0; i < cmdno; i++)	{
			SetCursor(menux+2, menuy+2+i);
			cout << (i+1) << ": " << sels[i].label;
		}
		SetCursor(menux+2, menuy+2+i);
		cout << MAXCOMMANDS << ": Return";

		// ----- get the user's selection
		SetCursor(menux, menuy+4+i);
		cout << "Select> ";
		cin >> selection;

		if (selection > 0 && selection <= cmdno)	{
			// --- call the menu selection function
			ClearScreen();
			cout << sels[selection-1].label << ':'
				<< endl << endl;
			(*sels[selection-1].menufunction)();
		}
		else if (selection == MAXCOMMANDS)
			break;	// exiting the menu
		else
			cout << '\a';
	}
}

// ---------------------------------
// character input 
// ---------------------------------
void GUI::UserInput(char& c, char *prompt)
{
	cout << endl << prompt << ": ";
	cout.flush();
	cin >> c;
}

// ---------------------------------
// integer input 
// ---------------------------------
void GUI::UserInput(int& i, char *prompt)
{
	cout << endl << prompt << ": ";
	cout.flush();
	cin >> i;
}

// ---------------------------------
// String input 
// ---------------------------------
void GUI::UserInput(String& s, char *prompt, int len)
{
	cout << endl << prompt << ": ";
	cout.flush();
	String str(len);
	cin >> str;
	s = str;
}

// ---------------------------------
// yes/no user interface
// ---------------------------------
Bool GUI::YesNo(char *question)
{
	char c = ' ';
	while (c != 'y' && c != 'n')	{
		UserInput(c, question);
		c = tolower(c);
		if (c != 'y' && c != 'n')
			cout << '\a';
	}
	cout.flush();
	return (Bool) (c == 'y');
}

// ---------------------------------
// error message
// ---------------------------------
void GUI::Error(char *message)
{
	cout << endl << '\a' << message << endl;
	AnyKey();
}

// ---------------------------------
// get a keyboard char
// ---------------------------------
char GUI::GetKBChar()
{
	int c;
	if (putback)	{
		c = putback;
		putback = 0;
	}
	else 	{
		c = _bios_keybrd(_KEYBRD_READ);
		if ((c & 0xff) == 0)
			c = (c >> 8) | 0x80;
		else
			c &= 0xff;
	}
	return (char) c;
}

// ---------------------------------
// put back a keyboard char
// ---------------------------------
void GUI::PutBack(char c)
{
	putback = c;
}

// ---------------------------------
// Display a status line
// ---------------------------------
void GUI::StatusLine(char *s)
{
	SetCursor(0, SCREENHEIGHT-1);
	cout << s;
	cout.flush();
	int len = strlen(s);
	while (len++ < SCREENWIDTH-1)
		cout << ' ';
	cout.flush();
}


// ---------------------------------
// Prompt for and read any key
// ---------------------------------
void GUI::AnyKey()
{
	cout << endl << "Any key..." << endl;
	GetKBChar();
}

// ---------------------------------
// Position the cursor
// ---------------------------------
void GUI::SetCursor(int x, int y)
{
	cout << "\033[" << (y+1) << ';' << (x+1) << 'H';
	cout.flush();
}

// ---------------------------------
// Clear the screen
// ---------------------------------
void GUI::ClearScreen()
{
	cout << "\033[2J";
	cout.flush();
}

// ---------------------------------
// Write a single character to the screen
// ---------------------------------
void GUI::WriteChar(char c, int x, int y)
{
	SetCursor(x, y);
	cout << c;
	cout.flush();
}

// ---------------------------------
// Test for a keyboard character waiting
// ---------------------------------
Bool GUI::KBCharWaiting()
{
	return (Bool) _bios_keybrd(_KEYBRD_READY);
}


