// ---------------- life.cpp

// ============================================
// The Game of Life - a simulation
// ============================================

#ifdef ZTC
#include <iostream.hpp>
#include <strstrea.hpp>
#include <iomanip.hpp>
#else
#include <iostream.h>
#include <strstrea.h>
#include <iomanip.h>
#endif
#include <ctype.h>
#include "gui.h"

#ifdef BCC
extern unsigned _stklen = 8192;
#endif
#ifdef ZTC
unsigned _stack = 8192;
#endif

Parody *life;

// ------- dimensions of the Life world
const int HT = SCREENHEIGHT-1;
const int WD = SCREENWIDTH;

const char Pop = 2;  // the populated display character

// ----- collection of neighboring cells
class Life : public Persistent {
	int world[WD][HT];
	int generation;
	Bool alive;
	int popcount;
	static int Rules[2][8];
	int CountNeighbors(int x, int y);
	void Read();
	void Write();
public:
	Life(ObjAddr objaddr = 0);
	~Life() { SaveObject(); }
	void Generation();
	Bool isAlive() { return alive; }
	void Populate(int x, int y);
	void DePopulate(int x, int y);
	Bool isPopulated(int x, int y)
		{ return (Bool) world[x][y]; }
};

// ------ read the persistent Life object
void Life::Read()
{
	int ct, x, y;
	ReadObject(ct);
	while (ct--)	{
		ReadObject(x);
		ReadObject(y);
		Populate(x,y);
	}
}

// ------ write the persistent Life object
void Life::Write()
{
	WriteObject(popcount);
	for (int x = 0; x < WD; x++)	{
		for (int y = 0; y < HT; y++)	{
			if (world[x][y])	{
				WriteObject(x);
				WriteObject(y);
			}
		}
	}
}

// ------ rules for cell survival
int Life::Rules[2][8] = {
	// --- cell is unpopulated
	{0,0,0,1,0,0,0,0},
	// --- cell is populated
	{0,0,1,0,0,0,0,0}
};

Life::Life(ObjAddr objaddr) : Persistent(*life, 0)
{
	for (int x = 0; x < WD; x++)
		for (int y = 0; y < HT; y++)
			world[x][y] = 0;
	alive = False;
	generation = 0;
	popcount = 0;
	LoadObject(objaddr);
}

// ---------- populate a cell
void Life::Populate(int x, int y)
{
	if (!world[x][y])
		popcount++;
	world[x][y] = 1;
	alive = True;
}

// ---------- depopulate a cell
void Life::DePopulate(int x, int y)
{
	if (world[x][y])
		--popcount;
	world[x][y] = 0;
}

// ------- count a cell's neighbors
int Life::CountNeighbors(int x, int y)
{
	int x1, y1, ct = 0;
	// ---- count 3 rows
	for (y1 = y-1; y1 < y+2; y1++)
		// ---- count 3 columns
		for (x1 = x-1; x1 < x+2; x1++)
			// --- don't count self
			if (!(y1 == y && x1 == x))
				// --- don't exceed margins
				if (y1 > -1 && y1 < HT && x1 > -1 && x1 < WD)
					// --- count populated neighbor cells
					if ((world[x1][y1]) & 1)
						ct++;
	return ct;
}

// -------- evolve one generation
void Life::Generation()
{
	gui.SetCursor(0, HT);
	char st[81];
	strstream stat(st, 80, ios::out);
	stat << "Generation: " << generation++ << '\0';
	gui.StatusLine(stat.str());
	int x, y, newpop, oldpop, nct, cell;
	for (x = 0; x < WD; x++)	{
		for (y = 0; y < HT; y++)	{
			nct = CountNeighbors(x, y);
			cell = world[x][y];
			oldpop = cell & 1;
			newpop = (Rules[oldpop][nct]) << 1;
			cell = oldpop | newpop;
			world[x][y] = cell;
		}
	}
	alive = False;
	popcount = 0;
	for (x = 0; x < WD; x++)	{
		for (y = 0; y < HT; y++)	{
			cell = world[x][y];
			newpop = cell >> 1;
			oldpop = cell & 1;
			if (newpop != oldpop)	{
				if (newpop)
					gui.WriteChar(Pop, x, y);
				else
					gui.WriteChar(' ', x, y);
			}
			if (newpop)	{
				alive = True;
				popcount++;
			}
			world[x][y] = newpop;
		}
	}
}

void BuildLife();
void DisplayLife();
void RunLife();
void StepLife();
void LoadLife();
void SaveLife();

Life *lf;

void main()
{
	life = new Parody("LIFE");
	gui.UserMenu( "The Game of Life",
				  "Build",   BuildLife,
				  "Display", DisplayLife,
				  "Run",     RunLife,
				  "Step",    StepLife,
				  "Load",    LoadLife,
				  "Save",    SaveLife,
				  (char*) NULL );
	delete lf;
}

// ------ create a new universe
void BuildLife()
{
	int x = 0, y = 0;
	unsigned char lastc = 0;
	unsigned char c = 0;
	delete lf;
	lf = new Life;
	gui.ClearScreen();
	gui.StatusLine("*, sp, Esc");
	while (c != ESC)	{
		gui.SetCursor(WD-12, HT);
		cout << "x:" << setw(3) << x;
		cout << " y:" << setw(3) << y;
		cout.flush();
		gui.SetCursor(x, y);
		c = gui.GetKBChar();
		switch (tolower(c))	{
			case ' ':
			case '\r':
				// --- toggle a population
				if (!lf->isPopulated(x,y))	{
					lf->Populate(x,y);
					gui.WriteChar(Pop, x, y);
				}
				else 	{
					lf->DePopulate(x,y);
					gui.WriteChar(' ', x, y);
				}
				gui.PutBack(lastc);
				break;
			case UPARROW:
				// --- up cursor
				if (y-- == 0)
					y = HT-1;
				lastc = c;
				break;
			case DOWNARROW:
				// --- down cursor
				if (++y == HT)
					y = 0;
				lastc = c;
				break;
			case LEFTARROW:
				// --- left cursor
				if (x-- == 0)
					x = WD-1;
				lastc = c;
				break;
			case RIGHTARROW:
				// --- right cursor
				if (++x == WD)
					x = 0;
				lastc = c;
				break;
			default:
				break;
		}
	}
}

// ----- display the universe
static void ShowWorld()
{
	gui.ClearScreen();
	if (lf != NULL)
		for (int y = 0; y < HT; y++)
			for (int x = 0; x < WD; x++)
				if (lf->isPopulated(x, y))
					gui.WriteChar(Pop, x, y);
}

// ----- display the universe and wait
void DisplayLife()
{
	ShowWorld();
	gui.AnyKey();
}

// ------ nonstop evolution
void RunLife()
{
	if (lf != NULL)	{
		ShowWorld();
		while (lf->isAlive())	{
			if (gui.KBCharWaiting())	{
				char c = gui.GetKBChar();
				if (c == ESC)
					break;
			}
			lf->Generation();
		}
	}
}

// --------- single-step evolution
void StepLife()
{
	if (lf != NULL)	{
		ShowWorld();
		gui.StatusLine("S-tep or Esc");
		int c;
		while ((c = gui.GetKBChar()) != ESC)
			if (tolower(c) == 's')
				lf->Generation();
	}
}

// -------- retrieve a persistent Life object
void LoadLife()
{
	int oa;
	gui.UserInput(oa, "Enter Life object #");
	delete lf;
	lf = new Life(oa);
	if (!lf->ObjectExists())
		gui.Error("Invalid object #");
	else
		DisplayLife();
}

// -------- store a persistent Life object
void SaveLife()
{
	if (lf != NULL)	{
		ObjAddr oa = lf->ObjectAddress();
		if (oa == 0)	{
			lf->AddObject();
			oa = lf->ObjectAddress();
		}
		else
			lf->ChangeObject();
		// ---- force the add/change now
		delete lf;
		lf = new Life(oa);
		cout << endl << "Life object #: " << oa;
		gui.AnyKey();
	}
}




