// -------- parody.h

#ifndef DATABASE_H
#define DATABASE_H

#ifdef ZTC
#include <fstream.hpp>
#else
#include <fstream.h>
#endif

enum Bool { False, True };

// ============================
// min/max macros
// ============================
inline int max(int a, int b)
{
	return a > b ? a : b;
}

inline int min(int a, int b)
{
	return a < b ? a : b;
}


#include "strings.h"
#include "btree.h"

// ============================
// Object Address
// ============================
typedef NodeNbr ObjAddr;

// ============================
// Persistent Object Header Rcd
// ============================
struct ObjectHeader	{
	int classid;		// invariant: class identification
	NodeNbr ndnbr;		// relative node number within object
	ObjectHeader(int cid = 0, NodeNbr nd = 0)
		{ classid = cid; ndnbr = nd; }
};

// ============================
// the Parody database
// ============================
class Parody : public FileHeader	{
	Index index;			   // invariant: the b-tree file
	LinkedListHead objects; // list of instantiated objects
	LinkedListHead btrees;	// the b-trees in the database
	ObjAddr rebuildnode;	   // node of object being rebuilt
public:
	Parody(String name);
	~Parody();
	Index& IndexFile()          { return index; }
	LinkedListHead& Btrees()    { return btrees; }
	LinkedListHead& Objects()   { return objects; }
	Bool RebuildingIndexes()
		{ return (Bool) (rebuildnode != 0); }
	ObjAddr RebuildNode()       { return rebuildnode; }
	void GetObjectHeader(ObjAddr nd, ObjectHeader &objhdr);
	void RebuildIndexes(ObjAddr nd);
};

// =====================================
// Persistent object abstract base class
// =====================================
class Persistent	{
	friend class Key;
	friend class ObjectList;
	ObjectHeader objhdr;
	ObjAddr objectaddress;	// Node address for this object
	Parody& parody;			// database for this object
	int indexcount;			// number of keys in the object
	LinkedListHead keys;	   // the keys in the object
	LinkedListHead orgkeys; // original keys in the object
	ObjectList *objectlist; // object list entry this object
	Node *node;				// current node for reading/writing
	int offset;				// current char position
	Bool changed;			// True if user changed the object
	Bool deleted;			// True if user deleted the object
	Bool newobject;		// True if user is adding the object
	// --- pointers to associate keys with objects
	Persistent *oldthispers;
	static Persistent *thispers;
	// ---- methods used from within Persistent class
	void ObjectOut();
	void RecordObject();
	void RemoveObject();
	void AddIndexes();
	void DeleteIndexes();
	void UpdateIndexes();
	void PositionNode();
	void ReadObjectHeader();
	void WriteObjectHeader();
	void SearchIndex(Key *key);
	void ReadDataMembers();
	Btree *FindIndex(Key *key);
	Bool TestRelationships();
	void ScanForward(NodeNbr nd);
	void ScanBackward(NodeNbr nd);
protected:
	Persistent(Parody &db, int cid);
	// ---- derived class methods for reading members
	void ReadObject(void *buf, int length);
	void ReadObject(String& str);
	void ReadObject(int& i)
		{ ReadObject(&i, sizeof(int)); }
	void ReadObject(long& l)
		{ ReadObject(&l, sizeof(long)); }
	void ReadObject(float& f)
		{ ReadObject(&f, sizeof(float)); }
	void ReadObject(double& d)
		{ ReadObject(&d, sizeof(double)); }
	void ReadObject(ObjAddr& o)
		{ ReadObject(&o, sizeof(ObjAddr)); }
	// ---- derived class methods for writing members
	void WriteObject(void *buf, int length);
	void WriteObject(String& str);
	void WriteObject(int i)
		{ WriteObject(&i, sizeof(int)); }
	void WriteObject(long l)
		{ WriteObject(&l, sizeof(long)); }
	void WriteObject(float f)
		{ WriteObject(&f, sizeof(float)); }
	void WriteObject(double d)
		{ WriteObject(&d, sizeof(double)); }
	void WriteObject(ObjAddr o)
		{ WriteObject(&o, sizeof(ObjAddr)); }
	// --- provided by derived class
	virtual void Write() = 0;
	virtual void Read() = 0;
public:
	virtual ~Persistent();
	// ---- called from derived class's constructor
	void LoadObject(ObjAddr nd = 0);
	// ---- called from derived class's destructor
	void SaveObject();
	// ---------- handle/copy variables
	int count;					// count of handle objects
	Persistent *objectcopy;	// for multiple instantiations
	// --- class interface methods for modifying database
	Bool AddObject();
	Bool ChangeObject();
	Bool DeleteObject();
	Bool ObjectExists()
		{ return (Bool) (objectaddress != 0); }
	// ---- class interface methods for searching database
	void FindObject(Key *key);
	void FirstObject(Key *key = NULL);
	void LastObject(Key *key = NULL);
	void NextObject(Key *key = NULL);
	void PreviousObject(Key *key = NULL);
	ObjAddr ObjectAddress() { return objectaddress; }
};

// ============================
// list of instantiated objects
// ============================
class ObjectList : public LinkedListEntry	{
	friend Persistent;
	Persistent *object;
	ObjectList(LinkedListHead *ll, Persistent *obj) :
			LinkedListEntry(ll)
		{ object = obj; }
};

// ==================================
// Handle class for handle/copy idiom
// ==================================
class Handle	{
protected:
	Persistent *body;
	Handle() { body = NULL; }
	virtual ~Handle();
	void ConstructBody(Persistent *pbody);
public:
	Handle(Handle& handle);
	Handle& operator=(Handle& empl);
	// --- class interface methods for modifying database
	Bool AddObject()     { return body->AddObject(); }
	Bool ChangeObject()  { return body->ChangeObject(); }
	Bool DeleteObject()  { return body->DeleteObject(); }
	Bool ObjectExists()  { return body->ObjectExists(); }
	// ---- class interface methods for searching database
	void FindObject(Key *key)
		{ body->FindObject(key); }
	void FirstObject(Key *key = NULL)
		{ body->FirstObject(key); }
	void LastObject(Key *key = NULL)
		{ body->LastObject(key); }
	void NextObject(Key *key = NULL)
		{ body->NextObject(key); }
	void PreviousObject(Key *key = NULL)
		{ body->PreviousObject(key); }
	ObjAddr ObjectAddress()
		{ return body->ObjectAddress(); }
};

#include "key.h"

#endif

