// ------------ personel.cpp

// ------------------------------------------------------
// A Personnel Application
// using PARODY, 
// the Persistent, (Almost) Relational Object Database
// management system
// ------------------------------------------------------

#include "personel.h"
#include "gui.h"

// ----------- Employee prototypes
void QueryEmployees();
void AddEmployee();
void ChangeEmployee();
void DeleteEmployee();
void ListEmployees();
void ListEmployeeProjects();
void CompareEmployees();

// ----------- Department prototypes
void QueryDepartments();
void AddDepartment();
void ChangeDepartment();
void DeleteDepartment();
void ListDepartments();
void ListDepartmentEmployees();
void ListAllDepartments();
void ListOneDepartment();

// ---------- Project prototypes
void QueryProjects();
void AddProject();
void ChangeProject();
void DeleteProject();
void ListProjects();
void ListProjectEmployees();

// ---------- Assignment prototypes
void QueryAssignments();
void AddAssignment();
void ChangeAssignment();
void DeleteAssignment();
void ListAssignments();
void PostAssignments();

#ifdef BCC
extern unsigned _stklen = 8192;
#endif
#ifdef ZTC
unsigned _stack = 8192;
#endif

// =====================================
// Personnel Database
// The Employee, Department, Project, and Assignment
// classes depend on this personnel pointer.
// =====================================
Parody *personnel;

void main()
{
	personnel = new Parody("PERSONEL");
		// ---- empty objects to declare relationships
	{
		Employee empl;    
		Department dept;  
		Project proj;     
		Assignment assgn;
	}

	gui.UserMenu( "Personnel Input",
				  "Employees",   QueryEmployees,
				  "Departments", QueryDepartments,
				  "Projects",    QueryProjects,
				  "Assignments", QueryAssignments,
				  (char*) NULL );
	delete personnel;
}

// -------------------------
// Employee processing
// -------------------------
void QueryEmployees()
{
	gui.UserMenu( "Employees",
				  "Add",      AddEmployee,
				  "Change",   ChangeEmployee,
				  "Delete",   DeleteEmployee,
				  "List",     ListEmployees,
				  "Compare",  CompareEmployees,
				  "Projects", ListEmployeeProjects,
				  (char*) NULL );
}

// ------------ add employee objects
void AddEmployee()
{
	Employee *employee;
	while ((employee = Employee::Get()) != NULL)	{
		if (employee->ObjectExists())	{
			Employee::Header();
			employee->Display();
			gui.Error("Employee already on file");
		}
		else	{
			employee->Input();
			if (!employee->AddObject())
				gui.Error("Add disallowed");
		}
		delete employee;
	}
}

// ------------ change an existing employee object
void ChangeEmployee()
{
	Employee *employee = Employee::GetExisting();
	if (employee != NULL)	{
		gui.AnyKey();
		employee->SelectChange();
		delete employee;
	}
}

// ------------ delete an existing employee object
void DeleteEmployee()
{
	Employee *employee = Employee::GetExisting();
	if (employee != NULL)	{
		if (gui.YesNo("Delete the employee record?"))
			if (!employee->DeleteObject())
				gui.Error("Delete disallowed");
		delete employee;
	}
}

// -------- list all the employees
void ListEmployees()
{
	Employee empl;
	empl.FirstObject();
	Employee::Header();
	while (empl.ObjectExists())	{
		empl.Display();
		empl.NextObject();
	}
	gui.AnyKey();
}

// --------- compare two employees
//    (demonstrates multiple-copy persistent objects
void CompareEmployees()
{
	Employee *employee1, *employee2;
	cout << "Compare Two Employees";
	cout.flush();
	if ((employee1 = Employee::GetExisting()) != NULL)	{
		if ((employee2 = Employee::GetExisting()) != NULL)	{
			Employee::Header();
			employee1->Display();
			employee2->Display();
			delete employee2;
		}
		delete employee1;
	}
	gui.AnyKey();
}

// ------- list the projects an employee is assigned to
void ListEmployeeProjects()
{
	Employee *employee = Employee::GetExisting();
	if (employee != NULL)	{
		Project::Header();
		Assignment assgn;
		assgn.SetEmplNo(employee->EmplNo());
		assgn.EmplKey().FindObject();
		while (assgn.ObjectExists() &&
					assgn.EmplNo() == employee->EmplNo())	{
			Project proj(assgn.ProjNo());
			if (proj.ObjectExists())
				proj.Display();
			assgn.NextObject((Key *)&assgn.EmplKey());
		}
		delete employee;
	}
	gui.AnyKey();
}

// -------------------------
// Department processing
// -------------------------
void QueryDepartments()
{
	gui.UserMenu( "Departments",
				  "Add",       AddDepartment,
				  "Change",    ChangeDepartment,
				  "Delete",    DeleteDepartment,
				  "List",      ListDepartments,
				  "Employees", ListDepartmentEmployees,
				  (char*) NULL );
}

// ---------- add department objects
void AddDepartment()
{
	Department *department;
	while ((department = Department::Get()) != NULL)	{
		if (department->ObjectExists())	{
			Department::Header();
			department->Display();
			gui.Error("Department already on file");
		}
		else	{
			department->Input();
			if (!department->AddObject())
				gui.Error("Add disallowed");
		}
		delete department;
	}
}

// ----------- change a department object
void ChangeDepartment()
{
	Department *department = Department::GetExisting();
	if (department != NULL)	{
		gui.AnyKey();
		department->SelectChange();
		delete department;
	}
}

// ----------- delete a department object
void DeleteDepartment()
{
	Department *department = Department::GetExisting();
	if (department != NULL)	{
		if (gui.YesNo("Delete the department record?"))
			if (!department->DeleteObject())
				gui.Error("Delete disallowed");
		delete department;
	}
}

// ---------- list the departments
void ListDepartments()
{
	Department dept;
	dept.FirstObject();
	Department::Header();
	while (dept.ObjectExists())	{
		dept.Display();
		dept.NextObject();
	}
	gui.AnyKey();
}

// --------- list employees in departments
void ListDepartmentEmployees()
{
	gui.UserMenu( "List Employees by Department",
				  "All departments", ListAllDepartments,
				  "One department",  ListOneDepartment,
				  (char*) NULL );
	gui.AnyKey();
}

// ------- list employees in all departments
void ListAllDepartments()
{
	Employee empl;
	Employee::Header();
	empl.DeptKey().FirstObject();
	while (empl.ObjectExists())	{
		empl.Display();
		empl.NextObject((Key *)&empl.DeptKey());
	}
	gui.AnyKey();
}

// ------- list employees in a selected department
void ListOneDepartment()
{
	Department *department;
	while ((department=Department::GetExisting()) != NULL) {
		int deptno = department->DeptNo();
		Employee::Header();
		Employee empl;
		empl.SetDeptNo(deptno);
		empl.DeptKey().FindObject();
		while (empl.ObjectExists() && empl.DeptNo()==deptno) {
			empl.Display();
			empl.NextObject((Key *)&empl.DeptKey());
		}
		delete department;
	}
	gui.AnyKey();
}

// -------------------------
// Project processing
// -------------------------
void QueryProjects()
{
	gui.UserMenu( "Projects",
				  "Add",       AddProject,
				  "Change",    ChangeProject,
				  "Delete",    DeleteProject,
				  "List",      ListProjects,
				  "Employees", ListProjectEmployees,
				  (char*) NULL );
}

// ---------- a project objects
void AddProject()
{
	Project *project;
	while ((project = Project::Get()) != NULL)	{
		if (project->ObjectExists())	{
			Project::Header();
			project->Display();
			gui.Error("Project already on file");
		}
		else	{
			project->Input();
			if (!project->AddObject())
				gui.Error("Add disallowed");
		}
		delete project;
	}
}

// -------- change a project object
void ChangeProject()
{
	Project *project = Project::GetExisting();
	if (project != NULL)	{
		gui.AnyKey();
		project->SelectChange();
		delete project;
	}
}

// -------- delete a project object
void DeleteProject()
{
	Project *project = Project::GetExisting();
	if (project != NULL)	{
		if (gui.YesNo("Delete the project record?"))
			if (!project->DeleteObject())
				gui.Error("Delete disallowed");
		delete project;
	}
}

// -------------- list the project objects
void ListProjects()
{
	Project proj;
	proj.FirstObject();
	Project::Header();
	while (proj.ObjectExists())	{
		proj.Display();
		proj.NextObject();
	}
	gui.AnyKey();
}

// -------- list employees assigned to a project
void ListProjectEmployees()
{
	Project *project = Project::GetExisting();
	if (project != NULL)	{
		Employee::Header();
		Assignment assgn;
		assgn.SetProjNo(project->ProjNo());
		assgn.ProjKey().FindObject();
		while (assgn.ObjectExists() &&
					assgn.ProjNo() == project->ProjNo())	{
			Employee empl(assgn.EmplNo());
			if (empl.ObjectExists())
				empl.Display();
			assgn.NextObject((Key *)&assgn.ProjKey());
		}
		delete project;
	}
	gui.AnyKey();
}

// -----------------------
// Assignments
// The Assignments class is a "connector" file
// -----------------------
void QueryAssignments()
{
	gui.UserMenu( "Assignments",
				  "Add",        AddAssignment,
				  "Change",     ChangeAssignment,
				  "Delete",     DeleteAssignment,
				  "List",       ListAssignments,
				  "Post hours",	PostAssignments,
				  (char*) NULL );
}

// ----------- add an assignment object
void AddAssignment()
{
	for (;;)	{
		Assignment *assignment = Assignment::Get();
		if (assignment == NULL)
			break;
		if (assignment->ObjectExists())	{
			Assignment::Header();
			assignment->DetailDisplay();
			gui.Error("Assignment already on file");
		}
		else	{
			assignment->Input();
			assignment->AddObject();
		}
		delete assignment;
	}
}

// ------------ change an assignment object
void ChangeAssignment()
{
	Assignment *assignment = Assignment::GetExisting();
	if (assignment != NULL)	{
		gui.AnyKey();
		assignment->Input();
		delete assignment;
	}
}

// ---------- delete an assignment object
void DeleteAssignment()
{
	Assignment *assignment = Assignment::GetExisting();
	if (assignment != NULL)	{
		if (gui.YesNo("Delete the assignment?"))
			assignment->DeleteObject();
		delete assignment;
	}
}

// ----- post hours expended by the employee to the project
void PostAssignments()
{
	Assignment *assignment = Assignment::GetExisting();
	if (assignment != NULL)	{
		int hours;
		gui.UserInput(hours, "Hours expended");
		if (hours != 0)	{
			assignment->AddHours(hours);
			Project project(assignment->ProjNo());
			if (project.ObjectExists())
				project.AddHours(hours);
		}
		delete assignment;
	}
}

// -------- list the assignment objects
void ListAssignments()
{
	Assignment assgn;
	assgn.FirstObject();
	Assignment::Header();
	while (assgn.ObjectExists())	{
		assgn.DetailDisplay();
		assgn.NextObject();
	}
	gui.AnyKey();
}


