/*
 /  File: glob.c
 /  File-name wildcard pattern matching for GNU.
 /  Copyright (C) 1985, 1988, 1989, 1990
 /  Free Software Foundation, Inc.
 /
 /  original author one semi-anonymous RMS.
 /
 /  modified for djgpp by Douglas Clifton 2/15/92.
 /
 /  for original source find GNULIB.Z?? on the nets
 /  for questions E-mail to: 99clifto@lab.cc.wmich.edu
 */

#include <sys/types.h>
#include <sys/dirent.h>
#include <stdlib.h>
#include "gnulib.h"

#define reg register

static int glob_match_after_star(char *, char *);
static char **glob_dir_to_array(char *, char **);
  
/* Nonzero if '*' and '?' do not match an initial '.' for glob_filename. */

int noglob_dot_filenames = 1;

/* Return nonzero if PATTERN has any special globbing chars in it. */

int
glob_pattern_p(pattern)
    char *pattern;
{
    reg char *p = pattern;
    reg char c;
    while ((c = *p++) != '\0')  {
        switch (c)  {
            case '?':
            case '[':
            case '*':
                return 1;
            case '\\':
                if (*p++ == '\0')
                    return 0;
        }
    }
    return 0;
}

/*
 /  Match the pattern PATTERN against the string TEXT;
 /  return 1 if it matches, 0 otherwise.
 /
 /  A match means the entire string TEXT is used up in matching.
 /
 /  In the pattern string, `*' matches any sequence of characters, `?'
 /  matches any character, [SET] matches any character in the specified
 /  set, [!SET] matches any character not in the specified set.
 /
 /  A set is composed of characters or ranges; a range looks like
 /  character hyphen character (as in 0-9 or A-Z).
 /  [0-9a-zA-Z_] is the set of characters allowed in C identifiers.
 /  Any other character in the pattern must be matched exactly.
 /
 /  To suppress the special syntactic significance of any of `[]*?!-\',
 /  and match the character exactly, precede it with a `\'.
 /
 /  If DOT_SPECIAL is nonzero,
 /  `*' and `?' do not match `.' at the beginning of TEXT.
 */

int
glob_match(pattern, text, dot_special)
    char *pattern, *text;
    int dot_special;
{
    reg char *p = pattern, *t = text;
    reg char c;

    while ((c = *p++) != '\0')  {
        switch (c)  {
            case '?':
                if (*t == '\0' || (dot_special && t == text && *t == '.'))
                    return 0;
                else
                    ++t;
                break;
            case '\\':
                if (*p++ != *t++)
                    return 0;
                break;
            case '*':
                if (dot_special && t == text && *t == '.')
                    return 0;
                return glob_match_after_star (p, t);
            case '[':   {
                reg char c1 = *t++;
                int invert;
                if (c1 == '\0')
                    return 0;
                invert = (*p == '!');
                if (invert)
                    p++;
                c = *p++;
                while (1)   {
                    reg char cstart = c, cend = c;
                    if (c == '\\')  {
                        cstart = *p++;
                        cend = cstart;
                    }
                    if (cstart == '\0')
                        return 0;            /* Missing ']'. */
                    c = *p++;
                    if (c == '-')   {
                        cend = *p++;
                        if (cend == '\\')
                            cend = *p++;
                        if (cend == '\0')
                            return 0;
                        c = *p++;
                    }
                    if (c1 >= cstart && c1 <= cend)
                        goto match;
                    if (c == ']')
                        break;
                }
                if (!invert)
                    return 0;
                break;
                /* Skip the rest of the [...] construct that already matched. */
match:
                while (c != ']')    {
                    if (c == '\0')
                        return 0;
                    c = *p++;
                    if (c == '\0')
                        return 0;
                    if (c == '\\')
                    p++;
                }
                if (invert)
                    return 0;
                break;
            }
            default:
                if (c != *t++)
                    return 0;
        }
    }
    return *t == '\0';
}

/*
 /  Like glob_match, but match PATTERN against
 /  any final segment of TEXT.
 */

static int
glob_match_after_star(pattern, text)
    char *pattern, *text;
{
    reg char *p = pattern, *t = text;
    reg char c, c1;

    while ((c = *p++) == '?' || c == '*')   {
        if (c == '?' && *t++ == '\0')
            return 0;
    }
    if (c == '\0')
        return 1;
    if (c == '\\')
        c1 = *p;
    else
        c1 = c;
    --p;
    while (1)   {
        if ((c == '[' || *t == c1) && glob_match(p, t, 0))
            return 1;
        if (*t++ == '\0')
            return 0;
    }
}

/*
 /  Return a vector of names of files in directory DIR whose
 /  names match glob pattern PAT. The names are not in any
 /  particular order. Wildcards at the beginning of PAT do not
 /  match an initial period if noglob_dot_filenames is nonzero.
 /
 /  The vector is terminated by an element that is a null pointer.
 /
 /  To free the space allocated, first free the vector's elements,
 /  then free the vector.
 /
 /  Return NULL if cannot get enough memory to hold the pointer
 /  and the names.
 /
 /  Return -1 if cannot access directory DIR.
 /  Look in errno for more information.
 */

char **
glob_vector(pat, dir)
    char *pat, *dir;
{
    struct globval    {
        struct globval *next;
        char *name;
    };
    DIR *d;
    reg struct dirent *dp;
    struct globval *lastlink;
    reg struct globval *nextlink;
    reg char *nextname;
    u_int count;
    int lose;
    reg char **name_vector;
    reg u_int i;

    if ((d = opendir(dir)) == NULL)
        return (char **)-1;
    lastlink = NULL;
    count = lose = 0;
    /*
     / Scan the directory, finding all names that match.
     / For each name that matches, allocate a struct globval
     / on the stack and store the name in it. Chain those
     / structs together; lastlink is the front of the chain.
     */
    while (1) {
        if ((dp = readdir(d)) == NULL) break;
        if (glob_match(pat, dp->d_name, noglob_dot_filenames)) {
            nextlink = (struct globval *) alloca(sizeof (struct globval));
            nextlink->next = lastlink;
            i = dp->d_namlen + 1;
            if ((nextname = (char *) malloc(i)) == NULL)    {
                lose++;
                break;
            }
            lastlink = nextlink;
            nextlink->name = nextname;
            bcopy(dp->d_name, nextname, i);
            count++;
        }
    }
    closedir (d);
    if (!lose)  {
        name_vector = (char **) malloc((count + 1) * sizeof (char *));
        lose |= name_vector == NULL;
    }
    if (lose)   {                                 /* out of memory? */
        while (lastlink)    {           /* free the strings we have */
            free(lastlink->name);
            lastlink = lastlink->next;
        }
        return NULL;
    }
    /* Copy the name pointers from the linked list into the vector. */
    for (i = 0; i < count; ++i) {
        name_vector[i] = lastlink->name;
        lastlink = lastlink->next;
    }
    name_vector[count] = NULL;
    return name_vector;
}

/*
 /  Return a new array, replacing array, which is the
 /  concatenation of each string in array to DIR.
 /  Return NULL if out of memory.
 */

static char **
glob_dir_to_array(dir, array)
    char *dir, **array;
{
    reg u_int i, l;
    int add_slash = 0;
    char **result;

    if ((l = strlen(dir)) == 0)
        return array;
    if (dir[l - 1] != '/')
        add_slash++;
    for (i = 0; array[i] != NULL; i++)
        ;
    result = (char **) malloc((i + 1) * sizeof (char *));
    if (result == NULL)
        return NULL;
    for (i = 0; array[i] != NULL; i++)  {
        result[i] = (char *) malloc(1 + l + add_slash + strlen(array[i]));
        if (result[i] == NULL)
            return NULL;
        strcpy (result[i], dir);
        if (add_slash)
            result[i][l] = '/';
        strcpy (result[i] + l + add_slash, array[i]);
    }
    result[i] = NULL;
    /* Free the input array.  */
    for (i = 0; array[i] != NULL; i++)
        free (array[i]);
    free ((char *) array);
    return result;
}

/*
 /  Do globbing on PATHNAME.  Return an array of pathnames that match,
 /  marking the end of the array with a null-pointer as an element.
 /  If no pathnames match, then the array is empty (first element is null).
 /  If there isn't enough memory, then return NULL.
 /  If a file system error occurs, return -1; `errno' has the error code.
 /
 /  Wildcards at the beginning of PAT, or following a slash,
 /  do not match an initial period if noglob_dot_filenames is nonzero.
 */

char **
glob_filename(pathname)
    char *pathname;
{
    char **result;
    char *directory_name, *filename;
    u_int directory_len;
    u_int result_size = 1;

    if ((result = (char **) malloc(sizeof (char *))) == NULL)
        return NULL;
    result[0] = NULL;
    /* Find the filename. */
    if ((filename = rindex (pathname, '/')) == NULL) {
        filename = pathname;
        directory_name = "";
        directory_len = 0;
    }
    else    {
        directory_len = (filename - pathname) + 1;
        directory_name = (char *) alloca(directory_len + 1);
        bcopy(pathname, directory_name, directory_len);
        directory_name[directory_len] = '\0';
        ++filename;
    }
    /*
     /  If directory_name contains globbing characters,
     /  then we have to expand the previous levels.
     /  Use recursion.
     */
    if (glob_pattern_p(directory_name))    {
        char **directories;
        reg u_int i;
        if (directory_name[directory_len - 1] == '/')
            directory_name[directory_len - 1] = '\0';
        if ((directories = glob_filename(directory_name)) == NULL)
            goto memory_error;
        else if (directories == (char **) -1)
            return (char **) -1;
        else if (*directories == NULL)  {
            free ((char *) directories);
            return ((char **)-1);
	}
    /*  We have successfully globbed the preceding directory name.
     /  For each name in directories, call glob_vector on it and
     /  filename. Concatenate the results together.
     */
    for (i = 0; directories[i] != NULL; i++)    {
        char **temp_results = glob_vector(filename, directories[i]);
        if (temp_results == NULL)
            goto memory_error;
        else if (temp_results == (char **) -1)
        /* This filename is probably not a directory. Ignore it. */
            ;
        else    {
            char **array = glob_dir_to_array(directories[i], temp_results);
            reg u_int l = 0;
            while (array[l] != NULL)
                ++l;
            result = (char **) realloc(result,(result_size + l) * sizeof (char *));
            if (result == NULL)
                goto memory_error;
            for (l = 0; array[l] != NULL; ++l)
                result[result_size++ - 1] = array[l];
            result[result_size - 1] = NULL;
            free((char *)array);
	    }
	}
    /* Free the directories.  */
    for (i = 0; directories[i] != NULL; i++)
        free (directories[i]);
    free ((char *) directories);
    return result;
    }
    /* If there is only a directory name, return it. */
    if (*filename == '\0')  {
        result = (char **) realloc((char *)result, 2 * sizeof(char *));
        if (result == NULL)
            return NULL;
        result[0] = (char *) malloc(directory_len + 1);
        if (result[0] == NULL)
            goto memory_error;
        bcopy(directory_name, result[0], directory_len + 1);
        result[1] = NULL;
        return result;
    }
    else    {
        /*
         /  Otherwise, just return what glob_vector
         /  returns appended to the directory name.
         */
        char **temp_results = glob_vector(filename,
                                         (directory_len == 0 ? "." : directory_name));
        if (temp_results == NULL || temp_results == (char **) -1)
            return temp_results;
    return glob_dir_to_array(directory_name, temp_results);
    }
memory_error:
    if (result != NULL) {
        reg u_int i;
        for (i = 0; result[i] != NULL; ++i)
            free (result[i]);
        free ((char *) result);
    }
    return NULL;
}

#ifdef TEST

int
main(argc, argv)
    int argc;
    char **argv;
{
    char **value;
    int i, optind;

    for (optind = 1; optind < argc; optind++)   {
        value = glob_filename(argv[optind]);
        if (value == NULL)
            puts ("virtual memory exhausted");
        else if (value == (char **)-1)
            perror (argv[optind]);
        else    {
            for (i = 0; value[i] != NULL; i++)
                puts (value[i]);
        }
    }
    exit (0);
}

#endif /* TEST */
