/* cal.c - print calendar for one month or one year
 * compatible with unix cal command
 * version 2.8
 *
 * cal [-options] [[month] year]    (numerical month argument)
 * or
 * cal [-options] [month] [year]    (verbal month argument)
 *
 * cal (C) 1992 by Unicorn Research Corporation
 * inspired by an Amiga program by Gary L. Brant
 * Compile with MSDOS defined to activate the MSDOS features.

29 March 1993, Darrel Hankerson hankedr@mail.auburn.edu
 Modifications and enhancements for OS/2 (and others).
 New command line options using getopt:
  [--nodata] [--data-file=data-file]
 and color options (if compiling with -DUSE_COLOR for OS/2 or MSDOS)
  [--nocolor] [--color-file=color-file]
 */

#define MSDOS     1  /* DELETE if not MSDOS */
#define USE_COLOR 1  /* DELETE if not MSDOS or OS/2 or not using colors */

#ifdef USE_COLOR
#ifdef OS2
#define INCL_SUB
#include <os2.h>
#else
#include <dos.h> /* for int86() */
#endif
#endif

#include <stdio.h>
#include <io.h>   /* for isatty() */
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>

#include "getopt.h"

#ifdef __ZTC__                    /* make Zortech compatible with stricmp() */
#define stricmp(a,b) strcmpl(a,b) /* case-insensitive string comparison */
#endif

#define FUDGE1  1          /* needed to make day of week come out right */
#define FUDGE2  6          /* for old style (Julian) calendar */
#define MONWID  24         /* width of month */
#define YEARWID 3*MONWID-2 /* width of yearly calendar */
#define LINEWID 80         /* width of array line[] */
#define DAY_DESC  LINEWID-MONWID  /* width of special day descriptions */

#ifdef USE_COLOR           /* define display attribute possibilities */
#define FG_BLACK     0x00
#define FG_BLUE      0x01
#define FG_GREEN     0x02
#define FG_CYAN      0x03
#define FG_RED       0x04
#define FG_VIOLET    0x05
#define FG_ORANGE    0x06
#define FG_LTGRAY    0x07
#define FG_DKGRAY    0x08
#define FG_BRTBLUE   0x09
#define FG_BRTGREEN  0x0a
#define FG_BRTCYAN   0x0b
#define FG_BRTRED    0x0c
#define FG_BRTVIOLET 0x0d
#define FG_YELLOW    0x0e
#define FG_WHITE     0x0f
#define FG_FLASH     0x80
#define BG_BLACK     0x00
#define BG_BLUE      0x10
#define BG_GREEN     0x20
#define BG_CYAN      0x30
#define BG_RED       0x40
#define BG_VIOLET    0x50
#define BG_ORANGE    0x60
#define BG_WHITE     0x70

char dfmon_attr   = FG_BLACK|BG_GREEN,   /* video attribute for month name */
     dfwk_attr    = FG_BLUE|BG_CYAN,     /* attribute for weekday header */
     dfdy_attr    = FG_BLACK|BG_WHITE,   /* attribute for days */
     dfsun_attr   = FG_VIOLET|BG_WHITE,  /* attribute for sundays */
     dftoday_attr = FG_YELLOW|BG_BLUE,   /* attribute for current day */
     dfbk_attr    = FG_LTGRAY|BG_BLACK,  /* year calendar background */
     dfspday_attr = FG_BRTCYAN|BG_BLACK, /* special day description attr. */
     dfspdfl_attr = FG_BRTRED|FG_FLASH,  /* special day flasher */
     dftoday_attr2;                      /* this must be initialized later */

char *mon_attr, *wk_attr, *day_attr, *spclday_attr, *line_attr[6],
     far *video, far *tmpvideo;
#ifdef MSDOS
union REGS reg;
#endif
int crt;
char *color_file = "cal.col";
#endif

char *data_file = "cal.dat";

short days[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 },
    mdays[] = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
    multi = 0, thisday, thismonth, thisyear, discard_old = 0, europe = 0;
char *months[] = {
       "January ", "February ", "March ", "April ", "May ", "June ", "July ",
       "August " , "September ", "October ", "November ", "December " },
     monthline[] = "\t  ---\t\t\t  ---\t\t\t  ---",
     *monthpos[] = { monthline+3, monthline+11, monthline+19 },
     USdays[] =     " Su Mo Tu We Th Fr Sa",
     Europedays[] = " Mo Tu We Th Fr Sa Su",
     *dayline = USdays,
     *spcldesc[2],
     *homedir,        /* home directory from argv[0], may not work in unix */
     *line[6],        /* line buffer for body of calendar */
     *buf,            /* workspace memory into which pointers are assigned */
     *str;            /* temporary string workspace (80 characters) */

void printmonth (short, short);
short putd (register short);
void fixtab (short);
short weekday(short, short);
void putattr(char *, register short);
short specialdays(short, short);
FILE *efopen(char *, char *);
void usage(void);

void main(int argc, char *argv[])
{
   short i = 8*LINEWID + 2*DAY_DESC, k, m, y;
   time_t t;
   struct tm *lt;

#ifdef USE_COLOR
   FILE *colorfile;
   short fg[8], bg[8];
   i += 9*LINEWID+DAY_DESC;  /* adjust workspace for color enhancements */
   crt = isatty(fileno(stdout));  /* check if stdout is being redirected */
#endif

   if ((buf = (char *)malloc(i)) == NULL) {
      puts("Memory overflow");
      return;
      }
/* assign pointers into buf */
   for (i = 0; i < 6; i++) line[i] = &buf[i*LINEWID];  /* calendar body */
   str = line[5] + LINEWID;                            /* temp workspace */
   *(spcldesc[0] = str + LINEWID) = '\0';              /* description 0 */
   *(spcldesc[1] = spcldesc[0] + DAY_DESC) = '\0';     /* description 1 */
                                    /* descriptions 2-7 point into line[] */
#ifdef USE_COLOR                                           /* color attributes */
   line_attr[0] = spcldesc[1] + DAY_DESC;
   for (i = 1; i < 6; i++) line_attr[i] = line_attr[i-1] + LINEWID;
   mon_attr = line_attr[5] + LINEWID;
   wk_attr = mon_attr + LINEWID;
   day_attr = wk_attr + LINEWID;
   spclday_attr = day_attr + LINEWID;
#endif

   for (;;) {
      static struct option long_options[] = {
         {"nodata", 0, 0, 0},
         {"data-file", 1, 0, 0},
         {"future", 0, 0, 0},
         {"europe", 0, 0, 0},
#ifdef USE_COLOR
         {"nocolor", 0, 0, 0},
         {"color-file", 1, 0, 0},
#endif
         {0, 0, 0, 0} };
      int c, option_index = 0;

      c = getopt_long (argc, argv, "d", long_options, &option_index);
      if (c == EOF) break;
      if (!c)
         switch(option_index) {
            case 0:
               data_file = NULL; break;
            case 1:
               data_file = optarg; break;
            case 2:
               discard_old = 1; break;
            case 3:
               dayline = Europedays;  europe = 1;  break;
#ifdef USE_COLOR
            case 4:
               crt = 0; break;
            case 5:
               color_file = optarg; break;
#endif
            }
      else {
         usage();
         goto freebuf;  /* end of function */
         }
      }
   for (i = 1; i <= argc-optind; i++)
      argv[i] = argv[optind+i-1];
   argc = argc-optind + 1;  /* now the argument list is compatible with unix cal */

#ifdef USE_COLOR
/* attempt to read in file of colors */
   if (!crt || (colorfile = efopen(color_file, "r")) == NULL) goto contsetup;
   for (i = 0; i < 8; i++) {
      if (fgets(str, 80, colorfile) == NULL) break;
      if ((fg[i] = atoi(str)) == (bg[i] = atoi(&str[3]))) break;
      if (fg[i] > 15 || bg[i] > 15) break;
      }
   fclose(colorfile);
   if (i < 8) goto contsetup;
   dfmon_attr   = fg[0] | (bg[0] << 4);
   dfwk_attr    = fg[1] | (bg[1] << 4);
   dfdy_attr    = fg[2] | (bg[2] << 4);
   dfsun_attr   = fg[3] | (bg[3] << 4);
   dftoday_attr = fg[4] | (bg[4] << 4);
   dfbk_attr    = fg[5] | (bg[5] << 4);
   dfspday_attr = fg[6] | (bg[6] << 4);
   dfspdfl_attr = fg[7] | (bg[7] << 4);

/* initialize color attribute strings */
contsetup:
   memset (mon_attr, dfmon_attr, LINEWID);
   memset (wk_attr, dfwk_attr, LINEWID);
   memset (day_attr, dfdy_attr, YEARWID);
   memset (spclday_attr, dfspday_attr, DAY_DESC);
   spclday_attr[0] = dfspdfl_attr;
   for (i = MONWID-2; i <= MONWID+MONWID; i += MONWID) {
      memset (&mon_attr[i], dfbk_attr, 2);
      memset (&wk_attr[i], dfbk_attr, 2);
      memset (&day_attr[i], dfbk_attr, 2);
      memset (&day_attr[i - MONWID + 3 + (18*europe)], dfsun_attr, 2);
      }
   memset (&day_attr[i - MONWID + 3 + (18*europe)], dfsun_attr, 2);
   dftoday_attr2 = ((dfdy_attr & '\x70')|(dftoday_attr >> 4));
#ifdef MSDOS
   int86(0x11, &reg, &reg);  /* read the equipment list into reg. */
   video = (char far *)MK_FP(((reg.x.ax & 48) == 48) ? 0xb000 : 0xb800, 0);
#endif
#endif

/* find originating directory */

   homedir = argv[0];
   i = strlen(homedir);
   while (i >= 0 && homedir[i] != '/' && homedir[i] != '\\' && homedir[i] != ':')
      homedir[i--] = '\0';

/* now that all the setup is done, we can begin with the program proper */

   t = time(NULL);
   lt = localtime(&t);
   thisyear = y = lt->tm_year + 1900;
   thismonth = m = lt->tm_mon + 1;
   thisday = lt->tm_mday;
   puts("");                 /* first display a blank line */

switch (argc) {
   case 1:                   /* display current month if no arguments */
      fixtab (y);
      printmonth (m, y);
      break;

   case 2:
      if (isdigit(argv[1][0]) || (isdigit(argv[1][1]) && argv[1][0] == '-')) {
         multi = 1;
         fixtab (y = atoi(argv[1]));
         fputs("\t\t\t\t ", stdout);
         putd(y);               /* put year */
         puts("\n");
         for (m = 1; m < 13; m++) printmonth (m, y);
         break;
         }
      /* drop through to next case */

   case 3:
      m = atoi(argv[1]);
      if (strlen(argv[1]) >= 3) {
         for (i = 0; i < 12; i++) {
            strcpy(str, months[i]);
            str[3] = argv[1][3] = '\0';
            if (!stricmp(argv[1], str)) {
               m = i + 1;
               break;
               }
            }
         }
      if (!m) {
          usage();
          break;
          }
      if (argc == 3) {
         if (isdigit(argv[2][0]) || (isdigit(argv[1][1]) && argv[2][0] == '-'))
            y = atoi(argv[2]);
         else {
            usage();
            break;
            }
         }
      fixtab(y);
      printmonth(m, y);
      break;

   default:
      usage();
   }
freebuf:
   free(buf);
}


/* printmonth () - either prints an entire month at a time or multiplexes
 * a month into a buffer, dumping the buffer after the third call.
 */
void printmonth (short m, short y)
{
   register short first, last;
   register short index, p = 0;
   register char *ll;
   static short q = 0;
   short l, hilite = ((m == thismonth) && (y == thisyear));

   --m;
   if (multi) {
      q++;
      if (q > 3) q = 1;
      p = MONWID * (q - 1);             /* character position of line in buffer */
      (void) memcpy(monthpos[q-1], months[m], 3);       /* copy month name */
      if (q == 3) {
         puts(monthline);
         #ifdef USE_COLOR
         putattr(mon_attr, YEARWID);
         #endif
         for (index = 0; index < 2; ++index) {
            fputs(dayline, stdout);
            fputs("   ", stdout);
         }
         puts(dayline);
         #ifdef USE_COLOR
         putattr(wk_attr, YEARWID);
         #endif
      }
      #ifdef USE_COLOR   /* initialize attribute buffers for year display */
      for (l = 0; l < 6; l++) memcpy(line_attr[l], day_attr, LINEWID);
      #endif
   } else {
      q = 1;
      #ifdef USE_COLOR   /* initialize attribute buffers for month display */
      for (l = 0; l < 6; l++) {
         memcpy(line_attr[l], day_attr, MONWID-2);
         line_attr[l][MONWID-2] = dfspdfl_attr;
         memcpy(&line_attr[l][MONWID-1], spclday_attr, DAY_DESC);
         }
      memcpy(&mon_attr[MONWID-1], spclday_attr, DAY_DESC);
      memcpy(&wk_attr[MONWID-1], spclday_attr, DAY_DESC);
      mon_attr[MONWID-2] = wk_attr[MONWID-2] = dfspdfl_attr;
      #endif
   }

   if (p == 0) memset(line[0], ' ', 6*LINEWID);

   if (y == 1752 && m == 8) {   /* special case Sep. 1752 */
      line[0][p + 8] = '1';
      line[0][p + 11] = '2';
      first = 14;
      last = 16;
      index = 12;
   }
   else {
      short dow = weekday(m, y);        /* day of week for first day of month */
      first = 1;
      last = 7 - dow;
      index = 3 * dow;
   }

   for (l = 0; l < 6; ++l) {    /* loop thru month one week per line */
      ll = line[l] + p + 1;
      while (first <= last) {   /* for each day in week encode day of month */
         if (first >= 10) ll[index] = '0' + first / 10;
         ll[index+1] = '0' + first % 10;
         if (!multi && hilite && first == thisday) {  /* hilight today */
            #ifdef USE_COLOR
            line_attr[l][index+1] = line_attr[l][index+2] = dftoday_attr;
            line_attr[l][index] = line_attr[l][index+3] = dftoday_attr2;
            ll[index-1] = crt ? '\xde' : '>';
            ll[index+2] = crt ? '\xdd' : '<';
            #else
            ll[index-1] = '>';
            ll[index+2] = '<';
            #endif
         }
         index += 3;
         ++first;
      }
      last = (last + 7) > days[m] ? days[m] : last + 7;
      index = 0;
   }
   if (!multi) {
      short i, yw;
      char mtmp[] = "          ";
      l = specialdays(m+1, y);
      mtmp[l = (18 - (i = strlen(months[m]))) >> 1] = '\0';
      fputs(mtmp, stdout);
      fputs(months[m], stdout);                 /* put month name */
      yw = putd(y);                                     /* put year */
      if (l) {
         mtmp[l] = ' ';
         mtmp[MONWID - l - i - yw - 1] = '\0';
         fputs(mtmp, stdout);
         puts(spcldesc[0]);                     /* put first description */
         #ifdef USE_COLOR
         putattr(mon_attr, LINEWID-1);
         #endif
      } else puts("");
      fputs(dayline, stdout);
      if (l > 1) {
         fputs("  ", stdout);
         puts(spcldesc[1]);                     /* put next description */
         #ifdef USE_COLOR
         putattr(wk_attr, LINEWID-1);
         #endif
      } else puts("");
   }
   for (l = 0; l < 6; l++)
      if (!multi || q == 3) {
         index = LINEWID-1;
         ll = line[l];
         while (index >= 0 && ll[index] == ' ') --index;
         ll[index+1] = '\0';
         puts(ll);
         #ifdef USE_COLOR
         putattr(line_attr[l], multi ? YEARWID : LINEWID-1);
         #endif
      }
}


/* putd - put year to standard output.
 */
short putd (register short n)
{
   register char *p = str+10;
   short yw = 0,
         neg = (n < 0);
   *p = '\0';
   if (neg) n = -n;
   do {
      --p;
      ++yw;
      *p = '0' + n % 10;
      n = n / 10;
   } while (n);
   if (neg) *(--p) = '-';
   fputs(p, stdout);
   return yw;
}


/* fixtab - correct for leapyears.
 */
void fixtab (short y)
{
   register short i;
   if (!(y % 4)) {
      if ((y % 100) || !(y % 400) || (y < 1753)) {
         days[1] = 29;
         for (i = 2; i < 13; i++) mdays[i]++;
      }
   }
}


/* weekday - return day-of-week for first day of month.
 */
short weekday (short m, short y)
{
   --y;
   if (y > 1752-1 || (y == 1752-1 && m > 8))
      return (mdays[m] + y + y / 4 - y / 100 + y / 400 + FUDGE1 - europe) % 7;
   else
      return (mdays[m] + y + y / 4                     + FUDGE2 - europe) % 7;
}


#ifdef USE_COLOR
void putattr(char *attr, register short len)  /* write video attr. string */
{
#ifdef OS2
   USHORT   row, col;
   int   i;
#endif
   if (!crt) return;
#ifdef OS2
   VioGetCurPos(&row, &col,0);
   row--;
   while (len) {        /* minimize the number of Vio calls */
      for (i = 1; i < len; i++)
         if (attr[i] != *attr) break;
      VioWrtNAttr((PBYTE) attr, i, row, col, 0);
      attr += i; col += i; len -= i;
   }
#else
   reg.h.ah = 3;
   reg.h.bh = 0;
   int86(0x10, &reg, &reg);
   tmpvideo = video + (160 * (reg.h.dh - 1) + reg.h.dl + 1);
   while (len--) {
      *tmpvideo = *(attr++);
      tmpvideo += 2;
      }
#endif
}
#endif


/* find the file CAL.DAT and attempt to read in special date descriptions.
*/
short specialdays(short m, short y)
{
FILE *fp;
short mo, dy, yr, w, n, wn, k, j, i = 1, item[8],
    hilight = (m == thismonth && y == thisyear);
char *lln[8], *ll;

if (!data_file) return 0;
if (m != thismonth) discard_old = 0;
lln[0] = spcldesc[0];
lln[1] = spcldesc[1];
for (j = 0; j < 6; j++) lln[2+j] = &line[j][MONWID-1];

*lln[0] = ' ';
lln[0][1] = '\0';
item[0] = 100;  /* day code for new month delimeter in descriptions */

if ((fp = efopen(data_file, "r")) == NULL) return 0;
while (i < 8 && fgets(str, 80, fp) != NULL) {   /* go until 8 done, or EOF */
   if (!isdigit(*str) && *str != '-') continue; /* a data line? */
   if ((yr = atoi(str)) == 0) continue;      /* get year */
   if ((mo = atoi(&str[5])) == 0) continue;  /* get month */
   if ((dy = atoi(&str[8])) == 0)                    /* get day */
      { if ((wn = atoi(&str[11])) == 0) continue; }  /* get nth wkday */
   else wn = 0;                   /* don't need wn if dy is valid */
   if ((yr != -999 && yr != y) || (mo != m && mo != (m%12)+1 && mo != -9)) /* got it all? */
      continue;
   w = wn/10;
   if (((wn >= 11 && wn <= 57) || w == 9) && (mo == m || mo == -9)) { /* find nth weekday */
      n = wn%10 - 1;              /* day of week (0=sun, 1=mon, etc.) */
      if (n < 0 || n > 6) continue;
      if (w == 9)                 /* special case to find last weekday */
         for (j = 5; j > 0; j--)
            { if (line[j][3*n+2] != ' ') break; }
      else {                      /* find nth weekday */
         for (j = 0; j < 6; j++)
            if (line[j][3*n+2] != ' ') if (!(--w)) break;
         if (w) continue;         /* nth weekday didn't exist */
         }
      dy = atoi(&line[j][3*n+1]); /* date of nth weekday */
      }
   if (!dy) continue;             /* next loop if wn>57 or wn<11 or dy=0 */
   if (mo == (m%12)+1) dy += 100; /* make sure 'next month' days are last */
   if (discard_old && dy < thisday) continue;  /* next loop if old day */
   item[i] = 0;
   for (j = 0; j < i; j++) {      /* insert in sorted order */
      if (dy < item[j]) {
         for (k = i; k > j; k--) {
            item[k] = item[k-1];
            strcpy(lln[k], lln[k-1]);
            }
         break;
         }
      }
   ll = lln[j];                   /* calendar line to modify */
   item[j] = dy;                  /* keep track of the dates so far */
   ll[0] = (hilight && dy == thisday && (mo == m || mo == -9)) ? '*' : ' ';  /* 'today' flag */
   ll[1] = (dy%100 < 10) ? ' ' : '0' + (dy%100)/10;
   ll[2] = '0' + dy%10;
   ll[3] = ':';
   str[j = 14+DAY_DESC-6] = '\0';
   j = strlen(str) - 1;           /* get rid of trailing white space */
   while (j >= 13 && isspace(str[j])) str[j--] = '\0';
   strcpy(&ll[4], &str[13]);      /* copy description into calendar */
   ++i;
   }
for (j = 0; j < i; j++) if (item[j] == 100) break;
if (j == i-1)
   lln[--i][0] = '\0';
else if (j < i) {
   strcat(lln[j], months[m%12]);
   strcat(lln[j], "--");
   }
return i;
}



FILE *efopen(char *file, char *mode)
{
   FILE  *fp;
#ifdef OS2
   char  path[_MAX_PATH];
#endif

   if ((fp = fopen(file, mode)) == NULL) {
      if (*homedir) {
         strcpy(str, homedir);
         strcat(str, file);
         fp = fopen(str, mode);
         }
#ifdef OS2
      if (fp == NULL) {
         _searchenv(file, "PATH", path);
         if (*path == '\0')
            _searchenv(file, "DPATH", path);
         if (*path)
            fp = fopen(path, mode);
         }
#endif
      }
   return fp;
}

void usage(void)
{
fputs("\n\
CAL 2.8 by Unicorn Research Corporation\n\
Display a monthly or yearly calendar, with optional appointments\n\n\
Usages:\tcal [options] [[month (1-12)] year]\n\
\tcal [options] [month (jan-dec)] [year]\n\n\
Options:\n\
  --nod[ata]\t\t   Ignore appointment descriptions file\n\
  --d[ata-file]=filename   Load appointments from `filename'\n\
  --f[uture]\t\t   Show only future appointments on current month\n\
  --e[urope]\t\t   European format (1st day is Monday)\n", stderr);
#ifdef USE_COLOR
fputs("\
  --noc[olor]\t\t   Inhibit use of colors\n\
  --c[olor-file]=filename  Load color definitions from `filename'\n", stderr);
#endif
}
