/*
 * Os9/68k fileio.c for MicroGnuEmacs by Robert A. Larson
 *	 system dependent file io routines
 */
#include "def.h"
#include "kbd.h"
#include <modes.h>
#include <dir.h>
#include <direct.h>

char	*getenv(), *index();

static	FILE	*ffp;

/*
 * Open a file for reading.
 */
ffropen(fn)
char	*fn;
{
	if ((ffp=fopen(fn, "r")) == NULL) return FIOFNF;
	return FIOSUC;
}

/*
 * Open a file for writing.
 */
ffwopen(fn)
char	*fn;
{
	if ((ffp=fopen(fn, "w")) == NULL) {
		ewprintf("Cannot open file %s for writing", fn);
		return FIOERR;
	}
	return FIOSUC;
}

/*
 * Close a file.
 * Should look at the status.
 */
ffclose()
{
	fclose(ffp);
	return FIOSUC;
}

/*
 * Write a buffer to the already
 * opened file. bp points to the
 * buffer. Return the status.
 * Check only at the newline and
 * end of buffer.
 */
ffputbuf(bp)
BUFFER *bp;
{
    register char *cp;
    register char *cpend;
    register LINE *lp;
    register LINE *lpend;

    lpend = bp->b_linep;
    lp = lforw(lpend);
    do {
	cp = &ltext(lp)[0];		/* begining of line	*/
	cpend = &cp[llength(lp)];	/* end of line		*/
	while(cp != cpend) {
	    putc(*cp, ffp);
	    cp++;	/* putc may evalualte arguments more than once */
	}
	lp = lforw(lp);
	if(lp == lpend) break;		/* no implied newline on last line */
	putc('\n', ffp);
    } while(!ferror(ffp));
    if(ferror(ffp)) {
	ewprintf("Write I/O error");
	return FIOERR;
    }
    return FIOSUC;
}

/*
 * Read a line from a file, and store the bytes
 * in the supplied buffer. Stop on end of file or end of
 * line.  When FIOEOF is returned, there is a valid line
 * of data without the normally implied \n.
 */
ffgetline(buf, nbuf, nbytes)
register char	*buf;
register int	nbuf;
register int	*nbytes;
{
	register int	c;
	register int	i;

	i = 0;
	while((c = getc(ffp))!=EOF && c!='\n') {
		buf[i++] = c;
		if (i >= nbuf) return FIOLONG;
	}
	if (c == EOF  && ferror(ffp) != FALSE) {
		ewprintf("File read error");
		return FIOERR;
	}
	*nbytes = i;
	return c==EOF ? FIOEOF : FIOSUC;
}

#ifndef NO_BACKUP
/*
 * Rename the file "fname" into a backup copy.
 * The backup copy is the same name with ".BAK" appended unless the file
 * name is to long.  The error handling is all in "file.c".
 */
fbackupfile(fname)
char	*fname;
{
	char *rindex();
	register char *backname, *cp;
	int stat;

	if((backname = malloc(strlen(fname) + 5)) == NULL) return FALSE;
/* delete the old backup */
	strcpy(backname, fname);
	strcat(backname, ".BAK");
	unlink(backname);			/* ignore errors */
	if(cp = rindex(fname, '/')) {
	    strcpy(backname, cp+1);
	    strcat(backname, ".BAK");
	}
	stat = rename(fname, backname) >= 0;
	free(backname);
	return stat;
}
#endif

#ifdef NO_BACKUP
#ifdef NO_DIRED
#define NO_RENAME
#endif
#endif

#ifndef NO_RENAME
rename(frname, toname)
char *frname, *toname;
{
	register char *params;
	register int frnamel, tonamel;
	int status;

	frnamel = strlen(frname);
	tonamel = strlen(toname);
	if((params = malloc(frnamel + tonamel + 2)) == NULL) return FALSE;
	strcpy(params, frname);
	params[frnamel] = ' ';
	strcpy(params + frnamel + 1, toname);
	if(os9fork("rename", frnamel + tonamel + 2, params, 0, 0, 0, 0)==-1) {
		free(params);
		return -1;
	}
	wait(&status);
	free(params);
	return (status & 0xffff)==0 ? 0 : -1;
}
#endif

/*
 * The string "fn" is a file name.
 * Perform any required appending of directory name or case adjustments.
 * If NO_DIR is not defined, the same file should be refered to even if the
 * working directory changes.  For Os9/68k, leave the file name case alone
 * so we use what the user specified.  (Should, but doesn't, use the case
 * stored on the disk for existing files.)
 */
#ifndef NO_DIR
extern char *wdir;
#endif

char *adjustname(fn)
register char	*fn;
{
    static char fnb[NFILEN];
    register char *cp;
    register char *rootp;

    switch(*fn) {
    	case '/':
	    cp = fnb;
	    *cp++ = *fn++;
	    while(*fn && (*cp++ = *fn++) != '/') {}
	    rootp = cp - 1;
	    break;
	case '~':
	    if(fn[1] != '/') return fn;		/* invalid, punt */
	    strcpy(fnb, getenv("HOME"));
	    rootp = index(fnb+1, '/');
	    cp = fnb + strlen(fnb);
	    if(cp==fnb) return fn;		/* invalid */
	    if(rootp == NULL) rootp = cp;
	    *cp++ = '/';
	    fn += 2;
	    break;
	default:
#ifndef	NODIR
	    strcpy(fnb, wdir);
	    rootp = index(fnb+1, '/');
	    cp = fnb + strlen(fnb);
	    if(cp==fnb) return fn;		/* invalid */
	    if(rootp == NULL) rootp = cp;
	    *cp++ = '/';
	    break;
#else
	    return fn;				/* punt */
#endif
    }
    while(*fn) {
    	if(*fn == '.') {
	    switch(fn[1]) {
	        case '\0':
		    *--cp = '\0';
		    return fnb;
	    	case '/':
	    	    fn += 2;
		    continue;
		case '.':
		    if(fn[2]=='/' || fn[2] == '\0') {
			--cp;
			while(cp > rootp && *--cp != '/') {}
			++cp;
			if(fn[2]=='\0') {
			    *--cp = '\0';
			    return fnb;
			}
		        fn += 3;
		        continue;
		    }
		    break;
		default: break;
	    }
	}
	while(*fn && (*cp++ = *fn++) != '/') {}
    }
    *cp = '\0';
    return fnb;
}

/*
 * fncmp: compare file or buffer names.  Return 0 on equality.  
 * (for compatibility with strcmp)  Both arguments have been 
 * through adjustname.
 */

fncmp(fna, fnb)
register char *fna, *fnb;
{
    register char ca, cb;

    while(ca = *fna++)
	if(ca != (cb = *fnb++) &&
		(!ISUPPER(ca) || TOLOWER(ca) != cb) &&
		(!ISUPPER(cb) || ca != TOLOWER(cb)))
	    return -1;
    return *fnb;
}

#ifndef NO_STARTUP
char *startupfile(suffix)
char *suffix;
{
	register char	*file;
	static char	home[NFILEN];
	char		*getenv();

	if ((file = getenv("HOME")) == NULL) goto notfound;
	if (strlen(file)+4 >= NFILEN - 1) goto notfound;
	(VOID) strcpy(home, file);
	(VOID) strcat(home, "/.mg");
	if (suffix != NULL) {
		(VOID) strcat(home, "-");
		(VOID) strcat(home, suffix);
	}
	if (access(home, 0) == 0) return home;

notfound:
#ifdef	STARTUPFILE
	file = STARTUPFILE;
	if (suffix != NULL) {
		(VOID) strcpy(home, file);
		(VOID) strcat(home, "-");
		(VOID) strcat(home, suffix);
		file = home;
	}
	if (access(file, 0) == 0) return file;
#endif

	return NULL;
}
#endif

#ifndef NO_DIR
char *getwd(cwd)
char *cwd;
{
    char backpath[MAXPATH];
    char *bpp = backpath, *path = cwd;
    DIR *dirp;
    struct direct *dp;
    long inode, inode2;
    char dots[MAXPATH];

    if((dirp = opendir(".")) == NULL || readdir(dirp) == NULL ||
		(dp = readdir(dirp)) == NULL) {
	closedir(dirp);
	return (char *)NULL;
    }
    inode = dp->d_addr;
    *path++ = '/';
    _gs_devn(dirp->dd_fd, path);
    path += strlen(path);
    closedir(dirp);
    strcpy(dots, "..");
    for(;;) {
	if((dirp = opendir(dots)) == NULL || readdir(dirp) == NULL ||
		    (dp = readdir(dirp)) == NULL) {
	    closedir(dirp);
	    return (char *)NULL;
	}
	inode2 = dp->d_addr;
	if(inode == inode2) break;
	do {
	    if((dp = readdir(dirp)) == NULL) {
		closedir(dirp);
		return (char *)NULL;
	    }
	} while(dp->d_addr != inode);
	*bpp++ = '/';
	strcpy(bpp, dp->d_name);
	bpp += strlen(bpp);
	closedir(dirp);
	inode = inode2;
	strcat(dots, "/..");
    }
    while(bpp > backpath) {
	*bpp = '\0';
	while(*--bpp != '/') {}
	strcpy(path, bpp);
	path += strlen(path);
    }
    return cwd;
}
#endif

#ifndef NO_DIRED
copy(frname, toname)
char *frname, *toname;
{
	register char *params;
	register int frnamel, tonamel;
	int status;

	frnamel = strlen(frname);
	tonamel = strlen(toname);
	if((params = malloc(frnamel + tonamel + 2)) == NULL) return FALSE;
	strcpy(params, frname);
	params[frnamel] = ' ';
	strcpy(params + frnamel + 1, toname);
	if(os9fork("copy", frnamel + tonamel + 2, params, 0, 0, 0, 0)==-1) {
		free(params);
		return -1;
	}
	wait(&status);
	free(params);
	return (status & 0xffff)==0 ? 0 : -1;
}

unlinkdir(fname)
char *fname;
{
    /* does NOT delete non-empty directories */
    if(_ss_attr(fname, S_IWRITE | S_IREAD) < 0) return -1;
    return unlink(fname);
}

BUFFER *dired_(dirname)
char *dirname;
{
    extern char *wdir;			/* defined in dir.c */
    register BUFFER *bp;
    char line[128];
    register FILE *dirpipe;
    int i;
    BUFFER *findbuffer();

    if(dirname == NULL || *dirname == '\0') dirname = wdir;
    else if((dirname = adjustname(dirname)) == NULL) {
	ewprintf("Could not adjust dir name");
	return NULL;
    }
    if((bp = findbuffer(dirname)) == NULL) {
	ewprintf("Could not create buffer");
	return NULL;
    }
    if(bclear(bp) != TRUE) return NULL;
    i = -1;
    strcpy(line, "-ae ");
    strcpy(&line[4], dirname);
	/* fopen doesn't work here for some reason, so use open & fdopen */
    if((i = open("/pipe", S_IREAD | S_IWRITE)) < 0 ||
    	    ((dirpipe = fdopen(i, "rw")) == NULL) || ((i = dup(1)) < 0) ||
	    (close(1) < 0) || (dup(fileno(dirpipe)) != 1) ||
	    (os9fork("dir", strlen(line) + 1, line, 0, 0, 0, 0) < 0) ||
	    (close(1) < 0) || (dup(i) != 1) || (close(i) < 0)) {
	ewprintf("Could not open pipe to dir");
	fclose(dirpipe);
	if(i >= 0) {
	    close(1);
	    dup(i);
	    close(i);
	}
	return NULL;
    }
    line[0] = line[1] = ' ';
    while(fgets(&line[2], 126, dirpipe) != NULL) {
    	line[strlen(line) - 1] = '\0';
	addline(bp, line);
    }
    if(fclose(dirpipe) < 0 || wait(&i) < 0 || i != 0) {
    	ewprintf("Trouble closing directory pipe");
	return NULL;
    }
    bp->b_dotp = lforw(bp->b_linep);		/* go to first line */
    strncpy(bp->b_fname, dirname, NFILEN);
    if((bp->b_modes[0] = name_mode("dired")) == NULL) {
	bp->b_modes[0] = &map_table[0];
	ewprintf("Could not find mode dired");
	return NULL;
    }
    bp->b_nmodes = 0;
    return bp;
}

d_makename(lp, fn)
register LINE *lp;
register char *fn;
{
    register char *cp;

    if(llength(lp) <= 54) return ABORT;
    strcpy(fn, curbp->b_fname);
    cp = fn + strlen(fn);
    *cp++ = '/';
    bcopy(&lp->l_text[54], cp, llength(lp) - 54);
    cp[llength(lp) - 54] = '\0';
    return lgetc(lp, 27) == 'd';
}
#endif
