/* Next available MSG number is  17 */

/*
 * Copyright (C) 1990 by Autodesk, Inc.
 * Copyright (C) 1992 by Antonio Costa, INESC-Norte.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 * ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR PURPOSE AND OF
 * MERCHANTABILITY ARE HEREBY DISCLAIMED.
 */

/*
 *   sol2scn.c - This program is written to produce realistic
 *		 solid images.
 *
 *		 It allows the user to select a solid. The
 *		 program then uses ap_q_solinfo() and walks
 *               thru the solid's CSG tree creating the
 *		 corresponding SCN tree.
 *		 A user specified file is created
 *               with "scn" extension. This program does
 *		 not do any particular viewing transformations.
 */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <adslib.h>
#include <adscodes.h>
#include <apsol.h>
#include <apfeat.h>
#include <approto.h>
#include "scn.h"

struct ads_comm
{
  char           *cstring;
  void            (*cfunc) ();
};

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#define NELEM(a) (sizeof a / sizeof a[0])

extern void     api_scn();
extern void     end_scn(), box2scn(), bool2scn(), sphere2scn();
extern void     cylinder2scn(), wedge2scn(), cone2scn(), rm2scn();
extern void     torus2scn();

struct ads_comm asolid_comm[] =
{
  { /* NT *//* MSG0 */ "C:SOL2SCN", api_scn},
};

static ap_Bool  funcload();

void
main(argc, argv)
  int             argc;
  char          **argv;
{
  int             stat;
  int             scode = RSRSLT;

  ads_init(argc, argv);
  for (;;)
  {
    if ((stat = ads_link(scode)) < 0)
    {
      ads_printf( /* NT *//* MSG1 */ "ERROR: bad status from ads_link = %d\n",
                 stat);
      fflush(stdout);
      exit(1);
    }
    scode = RSRSLT;		/* default return code */
    switch (stat)
    {
    case RQTERM:
      exit(0);
    case RQXLOAD:
      scode = funcload() ? RSRSLT : RSERR;
      break;
    case RQXUNLD:
      break;
    case RQSUBR:
      if ((stat = ads_getfuncode()) < 0)
      {
	ads_printf( /* NT *//* MSG2 */ "\nUnrecognized command.\n");
	ads_retvoid();
	break;
      }
      if (asolid_comm[stat].cfunc != (void (*) ()) NULL)
      {
	(*asolid_comm[stat].cfunc) ();
	ads_retvoid();
      }
      break;
    default:
      break;
    }
  }
}

/* FUNCLOAD -- Load external functions into AutoLISP */
static          ap_Bool
funcload()
{
  int             i;

  for (i = 0; i < NELEM(asolid_comm); i++)
  {
    if (ads_defun(asolid_comm[i].cstring, i) != RTNORM)
    {
      ads_printf( /* NT *//* MSG3 */ "Unable to register %s command.\n",
		 asolid_comm[i].cstring);
      return FALSE;
    }
#ifdef DEBUG
    ads_printf( /* NT *//* MSG4 */ "Registered %s command.\n",
               asolid_comm[i].cstring);
#endif
  }
  return TRUE;
}
static ap_Bool  init_scn();
static short    solid_scn();

void
api_scn()
{
  long            sslen;
  ads_name        sset, ename;
  ap_Solid        sol;

  ads_ssget(NULL, NULL, NULL, NULL, sset);
  ads_sslength(sset, &sslen);
  if (sslen > 1 || sslen <= 0)
  {
    ads_printf( /* NT *//* MSG5 */ "Many or No solids selected.\n");
    return;
  }
  ads_ssname(sset, 0, ename);
  ap_name2sol(ename, &sol);
  if (sol != (ap_Solid) NULL)
  {
    /*
     * set the cameras and the light sources
     */
    if (!init_scn())
      return;
    /*
     * now define the solid using scn commands recursively.
     */
    if (!solid_scn(sol))
    {
      ads_printf( /* NT *//* MSG6 */ "Solid SCN file generation failed.\n");
      return;
    }
  } else
  {
    ads_printf( /* NT *//* MSG7 */ "ERROR: ap_name2sol failed\n");
    return;
  }
  end_scn();
}

static void	scn_view();

static          ap_Bool
init_scn()
{
  struct resbuf   rb;
  char            prompt[50], fname[125], nfname[125];
  ads_point       vc, vd;
  int             i;
  RtPoint	  eye, at, up, dir;
  float		  fov;

  ads_getvar( /* NT *//* MSG0 */ "DWGNAME", &rb);
  sprintf(prompt, /* NT *//* MSG8 */ "File name <%s>: ", rb.resval.rstring);
  ads_getstring(FALSE, prompt, fname);
  if (strlen(fname) == 0)
    strcpy(fname, rb.resval.rstring);
  free(rb.resval.rstring);
  if (strcspn(fname, /* NT */ ".") == strlen(fname))
    strcat(fname, /* NT *//* MSG0 */ ".scn");
  if (ads_findfile(fname, nfname) == RTNORM)
  {
    char            res[80];

    ads_printf(\
/* NT *//* MSG9 */ "A SCN file with this name already exists.\n");
    res[0] = 0;
    ads_initget(0, /* NT *//* MSG10 */ "Yes No");
    ads_getkword( /* NT *//* MSG11 */ "Do you want to replace it? <N> ", res);
    if (res[0] != /* NT *//* MSG12 */ 'Y')
      return FALSE;
    strcpy(fname, nfname);
  }
  if (ads_getvar( /* NT *//* MSG0 */ "VIEWDIR", &rb) == RTNORM)
  {
    if (rb.restype == RT3DPOINT)
    {
      for (i = 0; i < 3; i++)
	vd[i] = rb.resval.rpoint[i];
    }
  }
  if (ads_getvar( /* NT *//* MSG0 */ "TARGET", &rb) == RTNORM)
  {
    if (rb.restype == RT3DPOINT)
    {
      for (i = 0; i < 3; i++)
	vc[i] = rb.resval.rpoint[i];
    }
  }
  ScnBegin(fname);
  scn_view(vd, vc, eye, at, &fov);
  up[0] = 0.0;
  up[1] = 0.0;
  up[2] = 1.0;
  ScnWorldBegin(eye, at, up, fov);
  dir[0] = -vd[0];
  dir[1] = -vd[1];
  dir[2] = -vd[2];
  ScnDistantLightSource(dir);
  return TRUE;
}

static void
end_scn()
{
  ScnWorldEnd();
  ScnEnd();
}

static short    rev2scn(), ext2scn();

static short
solid_scn(sol)
  ap_Solid        sol;
{
  ap_Solinfo      inf, *info;
  short           col, ret;
  RtMatrix	  mat;

  if (ap_q_solinfo(sol, &inf) != AP_NORMAL)
    goto solerr;
  info = &inf;
  col = info->s_color;
  rm2scn(info->s_mat, mat);
  switch (info->s_type)
  {
  case AP_BOX:
    box2scn(mat, info->s_param.box.x, info->s_param.box.y, info->s_param.box.z,
            col);
    break;
  case AP_CON:
    if (info->s_param.con.rx != info->s_param.con.ry)
      goto solerr;
    cone2scn(mat, info->s_param.con.rx, info->s_param.con.h, col);
    break;
  case AP_CYL:
    if (info->s_param.cyl.rx != info->s_param.cyl.ry)
      goto solerr;
    cylinder2scn(mat, info->s_param.cyl.rx, info->s_param.cyl.h, col);
    break;
  case AP_SPH:
    sphere2scn(mat, info->s_param.sph.r, col);
    break;
  case AP_TOR:
    if (info->s_param.tor.rmaj < 0.0)
      goto solerr;
    torus2scn(mat, info->s_param.tor.rmaj, info->s_param.tor.rmin, col);
    break;
  case AP_WED:
    wedge2scn(mat, info->s_param.wed.x, info->s_param.wed.y,
              info->s_param.wed.z, col);
    break;
  case AP_FIL:
    goto solerr;
  case AP_CHA:
    goto solerr;
  case AP_BOO:
    bool2scn(mat, info->s_param.boo.op1, info->s_param.boo.op2,
	     info->s_param.boo.op, col);
    break;
  case AP_EXT:
    if (info->s_param.ext.taper != 0.0)
      goto solerr;
    ret = ext2scn(mat, &(info->s_param.ext), col);
    if (ret < 0)
      goto solerr;
    break;
  case AP_REV:
    ret = rev2scn(mat, &(info->s_param.rev), col);
    if (ret < 0)
      goto solerr;
    break;
  }
  return 1;
solerr:
  ads_printf( /* NT *//* MSG13 */ "Unsupported primitive.\n");
  return 0;
}

static void	col2rgb();

/*
 * sphere2scn -- generate commands for sphere
 */
static void
sphere2scn(m, r, c)
  RtMatrix	  m;
  ads_real        r;
  short           c;
{
  static RtColor  col = {0.1, 0.3, 0.7};

  /* sphere */
  col2rgb(c, col);
  ScnSolidBegin(SCN_PRIMITIVE);
  ScnColor(col);
  ScnTransform(m);
  ScnSphere(r);
  ScnSolidEnd();
}

/*
 * cylinder2scn -- generate commands for cylinder
 */
static void
cylinder2scn(m, r, h, c)
  RtMatrix	  m;
  ads_real        r, h;
  short           c;
{
  static RtColor  col = {0.9, 0.0, 0.1};

  /* cylinder */
  col2rgb(c, col);
  ScnSolidBegin(SCN_LIST);
  ScnColor(col);
  ScnTransform(m);
  ScnClosedCylinder(h, r);
  ScnSolidEnd();
}

/*
 * cone2scn -- generate commands for cone
 */
static void
cone2scn(m, r, h, c)
  RtMatrix	  m;
  ads_real        r, h;
  short           c;
{
  static RtColor  col = {0.1, 0.8, 0.0};

  /* cone */
  col2rgb(c, col);
  ScnSolidBegin(SCN_LIST);
  ScnColor(col);
  ScnTransform(m);
  ScnClosedCone(h, r);
  ScnSolidEnd();
}

#define CP_PT(a, x, y, z) {a[0] = x; a[1] = y; a[2] = z;}

/*
 * box2scn -- generate commands for block
 */
static void
box2scn(m, x, y, z, c)
  RtMatrix	  m;
  ads_real        x, y, z;
  short           c;
{
  static RtColor  col = {0.9, 0.9, 0.0};

  /* box */
  col2rgb(c, col);
  ScnSolidBegin(SCN_PRIMITIVE);
  ScnColor(col);
  ScnTransform(m);
  ScnBox(x, y, z);
  ScnSolidEnd();
}

/*
 * wedge2scn -- generate commands for wedge
 */
static void
wedge2scn(m, x, y, z, c)
  RtMatrix	  m;
  ads_real        x, y, z;
  short           c;
{
  RtPoint         pt[4];
  static RtColor  col = {0.0, 0.9, 0.9};

  col2rgb(c, col);
  ScnSolidBegin(SCN_LIST);
  ScnColor(col);
  ScnTransform(m);
  CP_PT(pt[0], 0.0, 0.0, 0.0);
  CP_PT(pt[1], 0.0, y, 0.0);
  CP_PT(pt[2], x, y, 0.0);
  CP_PT(pt[3], x, 0.0, 0.0);
  ScnPolygon(4, (RtPointer) pt);
  CP_PT(pt[0], x, 0.0, 0.0);
  CP_PT(pt[1], x, y, 0.0);
  CP_PT(pt[2], x, 0.0, z);
  ScnPolygon(3, (RtPointer) pt);
  CP_PT(pt[0], 0.0, 0.0, 0.0);
  CP_PT(pt[1], 0.0, 0.0, z);
  CP_PT(pt[2], 0.0, y, 0.0);
  ScnPolygon(3, (RtPointer) pt);
  CP_PT(pt[0], x, y, 0.0);
  CP_PT(pt[1], 0.0, y, 0.0);
  CP_PT(pt[2], 0.0, 0.0, z);
  CP_PT(pt[3], x, 0.0, z);
  ScnPolygon(4, (RtPointer) pt);
  CP_PT(pt[0], 0.0, 0.0, 0.0);
  CP_PT(pt[1], x, 0.0, 0.0);
  CP_PT(pt[2], x, 0.0, z);
  CP_PT(pt[3], 0.0, 0.0, z);
  ScnPolygon(4, (RtPointer) pt);
  ScnSolidEnd();
}

/*
 * torus2scn -- generate commands for torus
 */
static void
torus2scn(m, maj, min, c)
  RtMatrix	  m;
  ads_real        maj, min;
  short           c;
{
  static RtColor  col = {0.9, 0.9, 0.9};

  col2rgb(c, col);
  ScnSolidBegin(SCN_LIST);
  ScnColor(col);
  ScnRotate(90.0, 1.0, 0.0, 0.0);
  ScnTransform(m);
  ScnTorus(maj, min);
  ScnSolidEnd();
}

/*
 * rev2scn -- generate commands for revolved sweeps
 */
static short
rev2scn(m, rev, c)
  RtMatrix	  m;
  struct aprevinfo *rev;
  short           c;
{
  static RtColor  col = {0.9, 0.9, 0.9};
  long            n;
  ap_Swp_pts     *rpt;
  RtPoint        *pt, hpt1, hpt2;
  long            i;

  n = rev->n;
  pt = (RtPoint *) malloc(n * sizeof(RtPoint));
  if (pt == (RtPoint *) NULL)
    return -1;
  /*
   * Autosolid always stores the data as if it revolved about
   * the Y axis.
   */
  col2rgb(c, col);
  ScnSolidBegin(SCN_LIST);
  ScnSolidBegin(SCN_PRIMITIVE);
  ScnColor(col);
  ScnRotate(-90.0, 1.0, 0.0, 0.0);
  ScnTransform(m);
  rpt = rev->pts;
  for (i = 0L; i < n; i++)
  {
    CP_PT(hpt1, rpt[i].x, 0.0, rpt[i].y);
    if (i == (n - 1L))
    {
      CP_PT(hpt2, rpt[0].x, 0.0, rpt[0].y);
    } else
    {
      CP_PT(hpt2, rpt[i + 1].x, 0.0, rpt[i + 1].y);
    }
    if (EQ(hpt1[0], hpt2[0]) && EQ(hpt1[0], 0.0))
      continue;
    ScnHyperboloid(hpt1, hpt2, rev->ang);
  }
  for (i = 0L; i < n; i++)
  {
    pt[i][0] = rpt[i].x;
    pt[i][1] = 0.0;
    pt[i][2] = rpt[i].y;
  }
  ScnPolygon(n, (RtPointer) pt);
  ScnSolidEnd();
  ScnSolidBegin(SCN_PRIMITIVE);
  ScnColor(col);
  ScnRotate(-90.0, 1.0, 0.0, 0.0);
  ScnRotate(rev->ang, 0.0, 0.0, 1.0);
  ScnTransform(m);
  for (i = n - 1; i >= 0L; i--)
  {
    pt[i][0] = rpt[n - 1 - i].x;
    pt[i][1] = 0.0;
    pt[i][2] = rpt[n - 1 - i].y;
  }
  ScnPolygon(n, (RtPointer) pt);
  ScnSolidEnd();
  ScnSolidEnd();
  free(pt);
  return 1;
}

static RtPoint *new_polygon();
/*
 * ext2scn -- generate commands for extruded sweeps
 */
static short
ext2scn(m, ext, c)
  RtMatrix	  m;
  struct apextinfo *ext;
  short           c;
{
  static RtColor  col = {0.9, 0.9, 0.9};
  long            n;
  ap_Swp_pts     *spt;
  RtPoint        *ept, *pt, fpt[4], *pt2;
  long            i, j;
  float           tmp;

  n = ext->n;
  spt = ext->pts;
  pt = new_polygon(spt, &n);
  if (pt == NULL)
    return -1;
  /*
   * first endcap (arcs not implemented)
   */
  col2rgb(c, col);
  ScnSolidBegin(SCN_LIST);
  ScnColor(col);
  ScnTransform(m);
  ScnPolygon(n, (RtPointer) pt);
  for (ept = pt, i = 0L; i < n; i++)
  {
    CP_PT(fpt[0], ept[i][0], ept[i][1], 0.0);
    CP_PT(fpt[1], ept[i][0], ept[i][1], ext->h);
    if (i == n - 1)
    {
      CP_PT(fpt[2], ept[0][0], ept[0][1], ext->h);
      CP_PT(fpt[3], ept[0][0], ept[0][1], 0.0);
    } else
    {
      CP_PT(fpt[2], ept[i + 1][0], ept[i + 1][1], ext->h);
      CP_PT(fpt[3], ept[i + 1][0], ept[i + 1][1], 0.0);
    }
    ScnPolygon(4, (RtPointer) fpt);
  }
  /*
   * second endcap (arcs not implemented)
   */
  pt2 = (RtPoint *) malloc(n * sizeof(RtPoint));
  if (pt2 == (RtPoint *) NULL)
    return -1;
  j = 2;
  for (i = 0L; i < n; i++)
  {
    pt2[i][0] = pt[j][0];
    pt2[i][1] = pt[j][1];
    pt2[i][2] = ext->h;
    j--;
    if (j < 0)
      j = n - 1;
  }
  ScnPolygon(n, (RtPointer) pt2);
  ScnSolidEnd();
  free(pt);
  free(pt2);
  return 1;
}

/*
 * rm2scn -- apply rigid motion to solid
 */
static void
rm2scn(mat1, mat2)
  ap_Trans3d      mat1;
  RtMatrix	  mat2;
{
  short           i, j;

  for (i = 0; i < 4; i++)
    for (j = 0; j < 4; j++)
      mat2[i][j] = mat1[i][j];
}

/*
 * bool2scn -- generate commands for boolean of two solids
 */
static void
bool2scn(m, soll, solr, op, c)
  RtMatrix	  m;
  ap_Solid        soll, solr;
  Ap_bool         op;
  short           c;
{
  static RtColor  col = {0.9, 0.2, 0.2};

  switch (op)
  {
  case AP_DIFF:
    ScnSolidBegin(SCN_DIFFERENCE);
    break;
  case AP_INT:
    ScnSolidBegin(SCN_INTERSECTION);
    break;
  case AP_UN:
    ScnSolidBegin(SCN_UNION);
    break;
  default:
    break;
  }
  /*****
  ScnSolidBegin(SCN_PRIMITIVE);
  col2rgb(c, col);
  ScnColor(col);
  ScnTransform(m);
  *****/
  solid_scn(soll);
  ScnSolidNext();
  solid_scn(solr);
  /*****
  ScnSolidEnd();
  *****/
  ScnSolidEnd();
}

static void
col2rgb(col, c)
  int             col;
  RtColor         c;
{
  static ads_real brightfac[5] = {1.0, 0.65, 0.5, 0.3, 0.15};
  ads_real        halfsat = .5;
  int             ih, vs;
  ads_real        h, s, f, value, r, g, b;

  r = g = b = 0.0;
  value = 1.0;

  switch (col)
  {
  case 0:
    break;
  case 1:
    r = 1.0;
    break;
  case 2:
    g = 1.0;
    b = 1.0;
    break;
  case 3:
    g = 1.0;
    break;
  case 4:
    g = 1.0;
    b = 1.0;
    break;
  case 5:
    b = 1.0;
    break;
  case 6:
    r = 1.0;
    b = 1.0;
    break;
  case 7:
  case 8:
  case 9:
    r = g = b = 1.0;
    break;
  default:			/* all the rest of the colors */
    if (col > 9 && col < 250)
    {
      /*
       * Apply the algorithm from Foley & van Dam to turn HSV into RGB values
       */
      ih = (col - 10) / 10;
      if (ih >= 24)
	ih -= 24;
      vs = col % 10;
      h = ih / 4.;
      ih = h;
      f = h - ih;
      value = brightfac[vs >> 1];
      s = vs & 1 ? halfsat : 1.0;
      switch (ih)
      {
      case 0:
	r = 1.0;
	g = (ads_real) (1.0 - s * (1.0 - f));
	b = (ads_real) (1.0 - s);
	break;
      case 1:
	r = (ads_real) (1.0 - s * f);
	g = 1.0;
	b = (ads_real) (1 - s);
	break;
      case 2:
	r = (ads_real) (1.0 - s);
	g = 1.0;
	b = (ads_real) (1.0 - s * (1.0 - f));
	break;
      case 3:
	r = (ads_real) (1.0 - s);
	g = (ads_real) (1.0 - s * f);
	b = 1.0;
	break;
      case 4:
	r = (ads_real) (1.0 - s * (1.0 - f));
	g = (ads_real) (1.0 - s);
	b = 1.0;
	break;
      case 5:
	r = 1.0;
	g = (ads_real) (1.0 - s);
	b = (ads_real) (1.0 - s * f);
	break;
      }
    } else
    {
      r = g = b = 1.0;
      value = 0.33 + (col - 250) * 0.134;
    }
    break;			/* Default */
  }
  c[0] = (r) * value;		/* Apply lightness scale factor */
  c[1] = (g) * value;		/* to components resulting from */
  c[2] = (b) * value;		/* hue and saturation. */
}

/*
 * scn_view -- calculate scene view parameters
 */
static void
scn_view(vd, vc, eye, at, fov)
  ads_point       vd, vc;
  RtPoint 	  eye, at;
  float		 *fov;
{
  at[0] = vc[0];
  at[1] = vc[1];
  at[2] = vc[2];
  eye[0] = vc[0] + vd[0];
  eye[1] = vc[1] + vd[1];
  eye[2] = vc[2] + vd[2];
  *fov = 22.5;
}

#define NEXT(x, n) ((x) == (n) - 1 ? 0 : (x) + 1)
#define PREV(x, n) ((x) == 0 ? (n) - 1 : (x) - 1)
#define ABSANG(x) (x < 0 ? x + AS2PI : x)
#define SPT(pt, x, y, z) pt[X] = (x); pt[Y] = (y); pt[Z] = (z)
#define ASPI  M_PI
#define AS2PI (2.0 * ASPI)

typedef struct pt_store
{
  ap_Swp_pts      pt[40];
  long            n;
  struct pt_store *p_next;
  struct pt_store *p_prev;
}               Pt_store;

/*
 * Calculate arc center point from 3-point arc
 */
void
arc2center(p1, p2, p3, cen)
  ads_point       p1, p2, p3, cen;
{
  ads_real        m1, m2, dp12x, dp12y, dp13x, dp13y;

  if ((p1[2] != p2[2]) || (p2[2] != p3[2]))
  {
    ads_printf( /* NT *//* MSG14 */ "ERROR: Z not same\n");
    return;
  }
  dp12x = p1[0] - p2[0];
  dp12y = p1[1] - p2[1];

  dp13x = p1[0] - p3[0];
  dp13y = p1[1] - p3[1];
  m1 = (-p2[1] * p2[1] + p1[1] * p1[1] - p2[0] * p2[0] + p1[0] * p1[0]) * 0.5;
  m2 = (-p3[1] * p3[1] + p1[1] * p1[1] - p3[0] * p3[0] + p1[0] * p1[0]) * 0.5;

  cen[1] = (-m2 * dp12x + m1 * dp13x) / (dp13x * dp12y - dp12x * dp13y);
  if (dp12x != 0.0)
    cen[0] = (m1 - dp12y * cen[1]) / dp12x;
  else
    cen[0] = (m2 - dp13y * cen[1]) / dp13x;
  cen[2] = p1[2];
}

static          ads_real
get_angle(ang1, ang2, ang3)
  ads_real        ang1, ang2, ang3;
{
  ap_Bool         cwise;
  ads_real        thet;

  cwise = FALSE;
  if (ang1 >= 0.0)
  {
    if ((ang1 > ang2) && (ang2 > ang3))
      cwise = TRUE;
    else if ((ang2 < 0.0) && (ang3 > ang1))
      cwise = TRUE;
    else if ((ang2 > 0.0) && (ang2 > ang3) && (ang3 > ang1))
      cwise = TRUE;
    else
      cwise = FALSE;
  } else
    /* if ang1 is negative */
  {
    if ((ang1 < ang2) && (ang2 < ang3))
      cwise = FALSE;
    else if ((ang2 > 0.0) && (ang3 < ang1))
      cwise = FALSE;
    else if ((ang2 < 0.0) && (ang2 < ang3) && (ang3 < ang1))
      cwise = FALSE;
    else
      cwise = TRUE;
  }

  if (cwise)
  {
    if (ang3 < ang1)
      thet = ang3 - ang1;
    else
      thet = ang3 - ang1 - AS2PI;
  } else
  {
    if (ang3 > ang1)
      thet = ang3 - ang1;
    else
      thet = ang3 - ang1 + AS2PI;
  }
  return thet;
}

/*
 * to create a new polygon consisting of only lines from the
 * original polygon made of arcs and lines.
 */
static          RtPoint * 
new_polygon(spts, n)
  ap_Swp_pts     *spts;
  long           *n;
{
  register        k, i, j;
  ads_real        thet, d_t;
  long            nn, newn, npts;
  RtPoint        *n_p;
  Pt_store       *p_pre, *tst2, *p_st = NULL, *tst;
  ads_point       n1, p1, p2, p3, cen;
  double          sin(), cos(), atan2(), sqrt();
  ads_real        r, alph1, alph2, alph3;

  npts = *n;
  nn = 0;
  for (i = 0; i < npts; i++)
  {
    if (spts[NEXT(NEXT(i, npts), npts)].type == AP_PT_ARCEND)
    {
      /*
       * circle made of 16 segments
       */
      d_t = AS2PI / (16.0);
      SPT(p1, spts[i].x, spts[i].y, 0.0);
      SPT(p2, spts[NEXT(i, npts)].x, spts[NEXT(i, npts)].y, 0.0);
      SPT(p3, spts[NEXT(NEXT(i, npts), npts)].x,
	  spts[NEXT(NEXT(i, npts), npts)].y, 0.0);
      arc2center(p1, p2, p3, cen);
      alph1 = atan2((p1[Y] - cen[Y]), (p1[X] - cen[X]));
      alph2 = atan2((p2[Y] - cen[Y]), (p2[X] - cen[X]));
      alph3 = atan2((p3[Y] - cen[Y]), (p3[X] - cen[X]));

      thet = get_angle(alph1, alph2, alph3);
      if (thet < 0.0)
	d_t = -1.0 * d_t;
      newn = (int) ((thet / d_t) + 0.5);
      if ((tst = (Pt_store *) malloc(sizeof(Pt_store))) == NULL)
      {
	ads_printf( /* NT *//* MSG15 */ "ERROR: Out of memory - new_polygon\n");
	for (tst2 = tst = p_st; tst2 != NULL; tst = tst2)
	{
	  tst2 = tst->p_next;
	  free(tst);
	}
	*n = 0;
	return (RtPoint *) NULL;
      }
      n1[X] = cen[X] - p1[X];
      n1[Y] = cen[Y] - p1[Y];
      n1[Z] = cen[Z] - p1[Z];
      r = sqrt(n1[X] * n1[X] + n1[Y] * n1[Y] + n1[Z] * n1[Z]);
      thet = d_t;
      for (j = 0; j < newn - 1; j++, thet = thet + d_t)
      {
	tst->pt[j].x = cen[X] + r * cos(alph1 + thet);
	tst->pt[j].y = cen[Y] + r * sin(alph1 + thet);
	nn = nn + 1;
      }
      tst->n = newn - 1;
      if (p_st == NULL)
      {
	p_pre = tst;
	tst->p_prev = NULL;
      } else
      {
	tst->p_prev = p_st->p_prev;
	p_st->p_prev = tst;
      }
      tst->p_next = p_st;
      p_st = tst;
      nn = nn - 1;
    }
    nn = nn + 1;
  }
  if ((n_p = (RtPoint *) malloc(sizeof(RtPoint) * nn)) == NULL)
  {
    ads_printf( /* NT *//* MSG16 */ "ERROR: Out of memory - new_polygon\n");
    for (tst2 = tst = p_st; tst2 != NULL; tst = tst2)
    {
      tst2 = tst->p_next;
      free(tst);
    }
    *n = 0;
    return (RtPoint *) NULL;
  }
  tst = p_pre;
  for (k = 0, i = 0; i < npts; i++, k++)
  {
    n_p[k][0] = spts[i].x;
    n_p[k][1] = spts[i].y;
    n_p[k][2] = 0.0;
    if (spts[NEXT(NEXT(i, npts), npts)].type == AP_PT_ARCEND)
    {
      for (j = 0; j < tst->n; j++)
      {
	k = k + 1;
	n_p[k][0] = tst->pt[j].x;
	n_p[k][1] = tst->pt[j].y;
	n_p[k][2] = 0.0;
      }
      tst = tst->p_prev;
      i = i + 1;
    }
  }
  for (tst2 = tst = p_st; tst2 != NULL; tst = tst2)
  {
    tst2 = tst->p_next;
    free(tst);
  }
  *n = nn;
  return n_p;
}
