/*
 * Copyright (c) 1988, 1992 Antonio Costa, INESC-Norte.
 * All rights reserved.
 *
 * Code, ideas or suggestions were taken from the following people:
 *
 *  Roman Kuchkuda      - basic ray tracer
 *  Mark VandeWettering - MTV ray tracer
 *  Augusto Sousa       - overall, shading model
 *  Reid Judd           - portability
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Antonio Costa, at INESC-Norte. The name of the author and
 * INESC-Norte may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#include "defs.h"
#include "extern.h"

/**********************************************************************
 *    RAY TRACING - Parameters - Version 8.0.0                        *
 *                                                                    *
 *    MADE BY    : Antonio Costa, INESC-Norte, October 1988           *
 *    ADAPTED BY : Antonio Costa, INESC-Norte, June 1989              *
 *    MODIFIED BY: Antonio Costa, INESC-Norte, September 1992         *
 **********************************************************************/

/***** Command line parameters *****/
#define THRESHOLD_COLOR(x) (3.0 * SQR(x))

static void
defaults()
{
  screen_size_x = 0;
  screen_size_y = 0;
  threshold_level = 0.01;
  threshold_color = THRESHOLD_COLOR(0.05);
  last_shade_level = 7;
  sampling_levels = 0;
  sampling_divisions = 1 SHL sampling_levels;
  sampling_weight = 4 * SQR(sampling_divisions);
  cluster_size = 4;
  last_ambient_level = 0;
  ambient_sample_rays = 16;
  antialiasing_mode = 0;
  background_mode = 0;
  raw_mode = 0;
  light_mode = 0;
  shade_mode = 1;
  normal_mode = 0;
  normal_check_mode = 0;
  texture_mode = 0;
  view_mode = 0;
  intersect_mode = 0;
  intersect_adjust_mode = 0;
  jittering_mode = 0;
  distributed_cache_mode = 0;
  focal_aperture = 0.0;
  focal_distance = 0.0;
  stereo_separation = 0.0;
  output_format = 0;    /* PIC */
#ifdef ECHO
  verbose_mode = 3;
#else
  verbose_mode = 2;
#endif
  contrast_mode = 0;

  lights_max = 16;
  objects_max = 0;
  surfaces_max = 256;
  csg_level_max = 256;
  movie_frames = 0;
}
static void
usage()
{
#ifndef lint
  extern char     copyright[];

  WRITE(ERROR, "%s", &(copyright[5]));
  FLUSH(ERROR);
  WRITE(ERROR, "Version %s created on %s %s.\n",
        PROGRAM_VERSION, __DATE__, __TIME__);
  FLUSH(ERROR);
#endif
  WRITE(ERROR, "Usage: {PROGRAM}\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [w]Screen Width       -> 16... (256)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [h]Screen Height      -> 16... (256)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [p]Sampling Levels    -> 0..%d (0)\n", SAMPLING_LEVEL_MAX);
  FLUSH(ERROR);
  WRITE(ERROR, "  [A]Aliasing Threshold -> 0..1 (0.05)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [s]Shading Levels     -> 1..%d (8)\n", last_shade_level);
  FLUSH(ERROR);
  WRITE(ERROR, "  [S]Shading Threshold  -> 0..1 (0.01) \n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [d]Ambient Levels     -> 0..%d (0)\n", last_shade_level);
  FLUSH(ERROR);
  WRITE(ERROR, "  [D]Ambient Samples    -> 2...\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [T]Ambient Threshold  -> 0..1 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [c]Cluster Size       -> %d..%d (4)\n",
        CLUSTER_SIZE_MIN, CLUSTER_SIZE_MAX);
  FLUSH(ERROR);
  WRITE(ERROR, "  [a]Antialiasing Mode  -> 0, 1, 2 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [i]Intersect Mode     -> 0, 1 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [I]Inter. Adjust Mode -> 0, 1 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [j]Jittering Mode     -> 0, 1 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [l]Lighting Mode      -> 0, 1, 2 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [m]Shading Mode       -> 0, 1 (1)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [n]Normal Mode        -> 0, 1 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [t]Texture Mode       -> 0, 1, 2 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [v]View Mode          -> 0, 1, 2 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [z]Normal Check Mode  -> 0, 1 (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [B]Mask Back File     -> FILE (only PIC format)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [R]Raw Picture File   -> FILE\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [F]Focal Distance     -> 0..%0.0f (GAZE)\n", X_MAX);
  FLUSH(ERROR);
  WRITE(ERROR, "  [E]Stereo Separation  -> 0..%0.0f\n", X_MAX);
  FLUSH(ERROR);
  WRITE(ERROR, "  [P]Focal Aperture     -> 0..1\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [O]Output Format      -> 0 - PIC format, 1 - PPM (0)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  [V]Verbose Mode       -> 0, 1, 2, 3 (2)\n");
  FLUSH(ERROR);
  WRITE(ERROR, "  {-|INFILE} {-|OUTFILE} [>LOGFILE]\n");
  FLUSH(ERROR);
}
#define MIN_PARAMETERS (2)

#define GET_INT(first, number)\
do {\
  IO_status = (sscanf(&(parameter[option][first]), "%hd", &(number)) == 1) ?\
    IO_OK : IO_READ;\
} while (0)

#define GET_LONG(first, number)\
do {\
  IO_status = (sscanf(&(parameter[option][first]), "%d", &(number)) == 1) ?\
    IO_OK : IO_READ;\
} while (0)

#define GET_REAL(first, number)\
do {\
  IO_status = (sscanf(&(parameter[option][first]), "%lf", &(number)) == 1) ?\
    IO_OK : IO_READ;\
} while (0)

static void
options()
{
  WRITE(results, "Image width        : %d\n", screen_size_x);
  FLUSH(results);
  WRITE(results, "Image height       : %d\n", screen_size_y);
  FLUSH(results);
  WRITE(results, "Aliasing threshold : %g\n", sampling_levels ?
        SQRT(threshold_color / 3.0) : 1.0);
  FLUSH(results);
  WRITE(results, "Shading threshold  : %g\n", last_shade_level ?
        threshold_level : 1.0);
  FLUSH(results);
  WRITE(results, "Ambient threshold  : %g\n", last_ambient_level ?
        1.0 - threshold_vector : 1.0);
  FLUSH(results);
  WRITE(results, "Sampling levels    : %d\n", sampling_levels);
  FLUSH(results);
  WRITE(results, "Shading levels     : %d\n", SUCC(last_shade_level));
  FLUSH(results);
  WRITE(results, "Ambient levels     : %d\n", last_ambient_level);
  FLUSH(results);
  WRITE(results, "Ambient samples    : %d\n", last_ambient_level ?
        ambient_sample_rays : 0);
  FLUSH(results);
  WRITE(results, "Cluster size       : %d\n", cluster_size);
  FLUSH(results);
  WRITE(results, "Antialiasing mode  : ");
  FLUSH(results);
  if (sampling_levels == 0)
  {
    WRITE(results, "PIXEL CORNER AVERAGE\n");
    FLUSH(results);
  }
  else
    switch (antialiasing_mode)
    {
    case 0:
      WRITE(results, "PIXEL ADAPTIVE SUPERSAMPLING\n");
      FLUSH(results);
      break;
    case 1:
      WRITE(results, "PIXEL SEMI-ADAPTIVE SUPERSAMPLING\n");
      FLUSH(results);
      break;
    case 2:
      WRITE(results, "PIXEL SUPERSAMPLING\n");
      FLUSH(results);
      break;
    }
  WRITE(results, "Background mode    : %s\n", background_mode ? "ON" : "OFF");
  FLUSH(results);
  WRITE(results, "Intersect mode     : %s\n",
        (intersect_mode OR(antialiasing_mode == 2)) ?
        "ALL OBJECTS" : "PIXEL CORNER OBJECTS");
  FLUSH(results);
  WRITE(results, "Inters. adjust mode: %s\n",
        intersect_adjust_mode ? "ON" : "OFF");
  FLUSH(results);
  WRITE(results, "Jittering mode     : %s\n", jittering_mode ? "ON" : "OFF");
  FLUSH(results);
  WRITE(results, "Lighting mode      : ");
  FLUSH(results);
  switch (light_mode)
  {
  case 0:
    WRITE(results, "NO TRANSLUCENT SHADOWS\n");
    FLUSH(results);
    break;
  case 1:
    WRITE(results, "PARTIAL TRANSLUCENT SHADOWS\n");
    FLUSH(results);
    break;
  case 2:
    WRITE(results, "FULL TRANSLUCENT SHADOWS\n");
    FLUSH(results);
    break;
  }
  WRITE(results, "Normal mode        : %s\n", normal_mode ?
        "NORMAL CORRECTION" : "NO NORMAL CORRECTION");
  FLUSH(results);
  WRITE(results, "Normal check mode  : %s\n", normal_check_mode ?
        "NORMAL TEXTURE CORRECTION" : "NO NORMAL TEXTURE CORRECTION");
  FLUSH(results);
  WRITE(results, "Shading mode       : %s\n", shade_mode ?
        "STRAUSS MODEL" : "PHONG MODEL");
  FLUSH(results);
  WRITE(results, "Raw mode           : %s\n", raw_mode ? "ON" : "OFF");
  FLUSH(results);
  WRITE(results, "Texture mode       : ");
  FLUSH(results);
  switch (texture_mode)
  {
  case 0:
    WRITE(results, "NO TEXTURES\n");
    FLUSH(results);
    break;
  case 1:
    WRITE(results, "TEXTURES INSIDE OBJECTS DESCRIPTION\n");
    FLUSH(results);
    break;
  case 2:
    WRITE(results, "TEXTURES AFTER OBJECTS DESCRIPTION\n");
    FLUSH(results);
    break;
  }
  WRITE(results, "View mode          : ");
  FLUSH(results);
  switch (view_mode)
  {
  case 0:
    WRITE(results, "NORMAL\n");
    FLUSH(results);
    break;
  case 1:
    WRITE(results, "LEFT EYE\n");
    FLUSH(results);
    break;
  case 2:
    WRITE(results, "RIGHT EYE\n");
    FLUSH(results);
    break;
  }
  if (focal_aperture > ROUNDOFF)
  {
    WRITE(results, "Focal aperture     : %g\n", focal_aperture);
    FLUSH(results);
  } else
  {
    WRITE(results, "Focal aperture     : %s\n", "0 - PINHOLE CAMERA");
    FLUSH(results);
  }
  if (focal_distance > ROUNDOFF)
  {
    WRITE(results, "Focal distance     : %g\n", focal_distance);
    FLUSH(results);
  } else
  {
    WRITE(results, "Focal distance     : %s\n", "GAZE");
    FLUSH(results);
  }
  if (stereo_separation > ROUNDOFF)
  {
    WRITE(results, "Stereo separation  : %g\n", stereo_separation);
    FLUSH(results);
  }
  if (stereo_separation < -ROUNDOFF)
  {
    WRITE(results, "Stereo separation  : %g%% of GAZE\n",
          -stereo_separation * 100.0);
    FLUSH(results);
  }
  WRITE(results, "Output format      : %s\n", output_format ? "PPM" : "PIC");
  FLUSH(results);
  WRITE(results, "Info: options ok\n");
  FLUSH(results);
}
void
get_parameters(parameters, parameter)
  int             parameters;   /* Command parameter count */
  char_ptr        parameter[];
{
  int             option;
  file_ptr        movie_eye, movie_look, movie_up, movie_angle;

  defaults();
  movie_eye = NULL;
  movie_look = NULL;
  movie_up = NULL;
  movie_angle = NULL;
  if (parameters < MIN_PARAMETERS)
  {
    WRITE(ERROR, "Error: PARAMETER(S) missing\n");
    FLUSH(ERROR);
    usage();
    HALT;
  }
  for (option = 0; option < parameters - MIN_PARAMETERS; POSINC(option))
  {
    switch (parameter[option][0])
    {
      case 'a':
        GET_INT(1, antialiasing_mode);
        if ((IO_status != IO_OK)
            OR(antialiasing_mode < 0) OR(antialiasing_mode > 2))
        {
          WRITE(ERROR, "Error: bad ANTIALIASING MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'c':
        GET_INT(1, cluster_size);
        if ((IO_status != IO_OK)
            OR(cluster_size < CLUSTER_SIZE_MIN)
            OR(cluster_size > CLUSTER_SIZE_MAX))
        {
          WRITE(ERROR, "Error: bad CLUSTER SIZE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'd':
        GET_INT(1, last_ambient_level);
        if ((IO_status != IO_OK)
            OR(last_ambient_level < 0)
            OR(last_ambient_level > last_shade_level))
        {
          WRITE(ERROR, "Error: bad AMBIENT LEVELS\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'h':
        GET_LONG(1, screen_size_y);
        if ((IO_status != IO_OK) OR(screen_size_y < 16))
        {
          WRITE(ERROR, "Error: bad SCREEN HEIGHT\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'i':
        GET_INT(1, intersect_mode);
        if ((IO_status != IO_OK)
            OR(intersect_mode < 0) OR(intersect_mode > 1))
        {
          WRITE(ERROR, "Error: bad INTERSECT MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'j':
        GET_INT(1, jittering_mode);
        if ((IO_status != IO_OK)
            OR(jittering_mode < 0) OR(jittering_mode > 1))
        {
          WRITE(ERROR, "Error: bad JITTERING MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'l':
        GET_INT(1, light_mode);
        if ((IO_status != IO_OK)
            OR(light_mode < 0) OR(light_mode > 2))
        {
          WRITE(ERROR, "Error: bad LIGHTING MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'm':
        GET_INT(1, shade_mode);
        if ((IO_status != IO_OK)
            OR(shade_mode < 0) OR(shade_mode > 1))
        {
          WRITE(ERROR, "Error: bad SHADING MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'n':
        GET_INT(1, normal_mode);
        if ((IO_status != IO_OK)
            OR(normal_mode < 0) OR(normal_mode > 1))
        {
          WRITE(ERROR, "Error: bad NORMAL MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'p':
        GET_INT(1, sampling_levels);
        if ((IO_status != IO_OK)
            OR(sampling_levels < 0) OR(sampling_levels > SAMPLING_LEVEL_MAX))
        {
          WRITE(ERROR, "Error: bad SAMPLING LEVEL(S)\n");
          FLUSH(ERROR);
          HALT;
        }
        sampling_divisions = 1 SHL sampling_levels;
        sampling_weight = 4 * SQR(sampling_divisions);
        break;

      case 's':
        GET_INT(1, last_shade_level);
        if ((IO_status != IO_OK) OR(last_shade_level <= 0))
        {
          WRITE(ERROR, "Error: bad SHADING LEVEL(S)\n");
          FLUSH(ERROR);
          HALT;
        }
        POSDEC(last_shade_level);
        break;

      case 't':
        GET_INT(1, texture_mode);
        if ((IO_status != IO_OK)
            OR(texture_mode < 0) OR(texture_mode > 2))
        {
          WRITE(ERROR, "Error: bad TEXTURE MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'v':
        GET_INT(1, view_mode);
        if ((IO_status != IO_OK)
            OR(view_mode < 0) OR(view_mode > 2))
        {
          WRITE(ERROR, "Error: bad VIEW MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'w':
        GET_LONG(1, screen_size_x);
        if ((IO_status != IO_OK) OR(screen_size_x < 16))
        {
          WRITE(ERROR, "Error: bad SCREEN WIDTH\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'z':
        GET_INT(1, normal_check_mode);
        if ((IO_status != IO_OK)
            OR(normal_check_mode < 0) OR(normal_check_mode > 1))
        {
          WRITE(ERROR, "Error: bad NORMAL CHECK MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'A':
        GET_REAL(1, threshold_color);
        if ((IO_status != IO_OK) OR(threshold_color <= -1.0)
            OR(threshold_color >= 1.0))
        {
          WRITE(ERROR, "Error: bad ALIASING THRESHOLD\n");
          FLUSH(ERROR);
          HALT;
        }
        if (threshold_color < 0.0)
          contrast_mode = 1;
        threshold_color = SQR(threshold_color) * 3.0;
        break;

      case 'B':
        if (background_mode == 1)
        {
          WRITE(ERROR, "Error: BACKGROUND MODE already set\n");
          FLUSH(ERROR);
          HALT;
        }
        OPEN(background, &(parameter[option][1]), WRITE_BINARY);
        if (IO_status != IO_OK)
        {
          WRITE(ERROR, "Error: unable to open BACKGROUND MASK FILE (%s)\n",
                &(parameter[option][1]));
          FLUSH(ERROR);
          HALT;
        }
        background_mode = 1;
        break;

      case 'D':
        GET_INT(1, ambient_sample_rays);
        if ((IO_status != IO_OK) OR(ambient_sample_rays < 2))
        {
          WRITE(ERROR, "Error: bad AMBIENT SAMPLES\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'E':
        GET_REAL(1, stereo_separation);
        if ((IO_status != IO_OK)
            OR(stereo_separation < -10.0) OR(stereo_separation > X_MAX))
        {
          WRITE(ERROR, "Error: bad STEREO SEPARATION\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'F':
        GET_REAL(1, focal_distance);
        if ((IO_status != IO_OK)
            OR(focal_distance < ROUNDOFF) OR(focal_distance > X_MAX))
        {
          WRITE(ERROR, "Error: bad FOCAL DISTANCE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'I':
        GET_INT(1, intersect_adjust_mode);
        if ((IO_status != IO_OK)
            OR(intersect_adjust_mode < 0) OR(intersect_adjust_mode > 1))
        {
          WRITE(ERROR, "Error: bad INTERSECT ADJUST MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'O':
        GET_INT(1, output_format);
        if ((IO_status != IO_OK)
            OR(output_format < 0) OR(output_format > 1))
        {
          WRITE(ERROR, "Error: bad OUTPUT FORMAT\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'P':
        GET_REAL(1, focal_aperture);
        if ((IO_status != IO_OK)
            OR(focal_aperture < ROUNDOFF) OR(focal_aperture > 1.0))
        {
          WRITE(ERROR, "Error: bad FOCAL APERTURE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'R':
        if (raw_mode == 1)
        {
          WRITE(ERROR, "Error: RAW MODE already set\n");
          FLUSH(ERROR);
          HALT;
        }
        OPEN(raw_picture, &(parameter[option][1]), WRITE_BINARY);
        if (IO_status != IO_OK)
        {
          WRITE(ERROR, "Error: unable to open RAW PICTURE FILE (%s)\n",
                &(parameter[option][1]));
          FLUSH(ERROR);
          HALT;
        }
        raw_mode = 1;
        break;

      case 'S':
        GET_REAL(1, threshold_level);
        if ((IO_status != IO_OK) OR(threshold_level <= 0.0)
            OR(threshold_level >= 1.0))
        {
          WRITE(ERROR, "Error: bad SHADING THRESHOLD\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case 'T':
        GET_REAL(1, threshold_vector);
        if ((IO_status != IO_OK) OR(threshold_vector < 0.0)
            OR(threshold_vector >= 1.0))
        {
          WRITE(ERROR, "Error: bad AMBIENT THRESHOLD\n");
          FLUSH(ERROR);
          HALT;
        }
        distributed_cache_mode = (threshold_vector > ROUNDOFF);
        break;

      case 'V':
        GET_INT(1, verbose_mode);
        if ((IO_status != IO_OK)
            OR(verbose_mode < 0) OR(verbose_mode > 3))
        {
          WRITE(ERROR, "Error: bad VERBOSE MODE\n");
          FLUSH(ERROR);
          HALT;
        }
        break;

      case '+':
        if (parameter[option][1] != EOT)
        {
          switch (parameter[option][1])
          {
            case 'C':
              GET_INT(2, csg_level_max);
              if ((IO_status != IO_OK) OR(csg_level_max < 1))
              {
                WRITE(ERROR, "Error: bad CSG LEVEL MAX\n");
                FLUSH(ERROR);
                HALT;
              }
              break;

            case 'L':
              GET_LONG(2, lights_max);
              if ((IO_status != IO_OK) OR(lights_max < 1))
              {
                WRITE(ERROR, "Error: bad LIGHTS MAX\n");
                FLUSH(ERROR);
                HALT;
              }
              break;

            case 'O':
              GET_LONG(2, objects_max);
              if ((IO_status != IO_OK) OR(objects_max < 1))
              {
                WRITE(ERROR, "Error: bad OBJECTS MAX\n");
                FLUSH(ERROR);
                HALT;
              }
              break;

            case 'S':
              GET_LONG(2, surfaces_max);
              if ((IO_status != IO_OK) OR(surfaces_max < 1))
              {
                WRITE(ERROR, "Error: bad SURFACES MAX\n");
                FLUSH(ERROR);
                HALT;
              }
              break;

            default:
              WRITE(ERROR, "Error: bad DIMENSION OPTION [+%c]\n",
                    parameter[option][1]);
              FLUSH(ERROR);
              usage();
              HALT;
              break;
          }
          break;
        }

      case 'M':
        if (parameter[option][1] != EOT)
        {
          switch (parameter[option][1])
          {
            case '+':
              GET_INT(2, movie_frames);
              if ((IO_status != IO_OK) OR(movie_frames < 1))
              {
                WRITE(ERROR, "Error: bad number of MOVIE FRAMES\n");
                FLUSH(ERROR);
                HALT;
              }
              break;

            case 'e':
              if (movie_eye != NULL)
              {
                WRITE(ERROR, "Error: MOVIE EYE FILE already set\n");
                FLUSH(ERROR);
                HALT;
              }
              OPEN(movie_eye, &(parameter[option][2]), READ_TEXT);
              if (IO_status != IO_OK)
              {
                WRITE(ERROR, "Error: unable to open MOVIE EYE FILE (%s)\n",
                      &(parameter[option][2]));
                FLUSH(ERROR);
                HALT;
              }
              if (movie_frames == 0)
                movie_frames = -1;
              break;

            case 'l':
              if (movie_look != NULL)
              {
                WRITE(ERROR, "Error: MOVIE LOOK FILE already set\n");
                FLUSH(ERROR);
                HALT;
              }
              OPEN(movie_look, &(parameter[option][2]), READ_TEXT);
              if (IO_status != IO_OK)
              {
                WRITE(ERROR, "Error: unable to open MOVIE LOOK FILE (%s)\n",
                      &(parameter[option][2]));
                FLUSH(ERROR);
                HALT;
              }
              if (movie_frames == 0)
                movie_frames = -1;
              break;

            case 'u':
              if (movie_up != NULL)
              {
                WRITE(ERROR, "Error: MOVIE UP FILE already set\n");
                FLUSH(ERROR);
                HALT;
              }
              OPEN(movie_up, &(parameter[option][2]), READ_TEXT);
              if (IO_status != IO_OK)
              {
                WRITE(ERROR, "Error: unable to open MOVIE UP FILE (%s)\n",
                      &(parameter[option][2]));
                FLUSH(ERROR);
                HALT;
              }
              if (movie_frames == 0)
                movie_frames = -1;
              break;

            case 'a':
              if (movie_angle != NULL)
              {
                WRITE(ERROR, "Error: MOVIE ANGLE FILE already set\n");
                FLUSH(ERROR);
                HALT;
              }
              OPEN(movie_angle, &(parameter[option][2]), READ_TEXT);
              if (IO_status != IO_OK)
              {
                WRITE(ERROR, "Error: unable to open MOVIE ANGLE FILE (%s)\n",
                      &(parameter[option][2]));
                FLUSH(ERROR);
                HALT;
              }
              if (movie_frames == 0)
                movie_frames = -1;
              break;

            default:
              WRITE(ERROR, "Error: bad MOVIE OPTION [M%c]\n",
                    parameter[option][1]);
              FLUSH(ERROR);
              usage();
              HALT;
              break;
          }
          break;
        }

      default:
        WRITE(ERROR, "Error: bad OPTION [%c]\n", parameter[option][0]);
        FLUSH(ERROR);
        usage();
        HALT;
        break;
    }
  }
  if ((screen_size_x == 0) AND(screen_size_y == 0))
  {
    screen_size_x = 256;
    screen_size_y = 256;
  } else
  if (screen_size_x == 0)
    screen_size_x = screen_size_y;
  else
  if (screen_size_y == 0)
    screen_size_y = screen_size_x;
  if ((view_mode == 0) AND(ABS(stereo_separation) > ROUNDOFF))
  {
    WRITE(ERROR, "Error: cannot have STEREO SEPARATION\n");
    FLUSH(ERROR);
    HALT;
  }
  if ((view_mode != 0) AND(ABS(stereo_separation) < ROUNDOFF))
  {
    WRITE(ERROR, "Error: no STEREO SEPARATION\n");
    FLUSH(ERROR);
    HALT;
  }
  if (threshold_color > THRESHOLD_COLOR(0.9))
  {
    sampling_levels = 0;
    sampling_divisions = 1 SHL sampling_levels;
    sampling_weight = 4 * SQR(sampling_divisions);
  }
  if (distributed_cache_mode != 0)
  {
    distributed_cache_repetitions = MAX(8, threshold_vector * screen_size_x);
    threshold_vector = 1.0 - threshold_vector;
  } else
    threshold_vector = 1.0;
  if (objects_max == 0)
    objects_max = 25000;
  else
    objects_max = (long) (objects_max * (cluster_size + 2.0) / cluster_size +
                          SUP(LOGN(objects_max, cluster_size))) - 1;
  if ((parameter[option][0] == '-') AND(parameter[option][1] == EOT))
    scene = INPUT;
  else
  {
    OPEN(scene, parameter[option], READ_TEXT);
    if (IO_status != IO_OK)
    {
      WRITE(ERROR, "Error: unable to open SCENE (%s)\n",
            parameter[option]);
      FLUSH(ERROR);
      HALT;
    }
  }
  POSINC(option);
  if ((parameter[option][0] == '-') AND(parameter[option][1] == EOT))
  {
#ifdef dos
    runtime_abort("cannot write correctly to OUTPUT");
#else
#ifdef THINK_C
    runtime_abort("cannot write correctly to OUTPUT");
#endif
#endif
    picture = OUTPUT;
    results = ERROR;
  } else
  {
    OPEN(picture, parameter[option], WRITE_BINARY);
    if (IO_status != IO_OK)
    {
      WRITE(ERROR, "Error: unable to open PICTURE (%s)\n",
            parameter[option]);
      FLUSH(ERROR);
      HALT;
    }
    results = OUTPUT;
  }
#ifdef THINK_C
  results = OUTPUT;
#endif

  if (movie_frames != 0)
  {
    int             i, movie_frames_max;

    if (movie_frames < 0)
      movie_frames = 50;
    movie_frames_max = movie_frames;
    ALLOCATE(camera, camera_struct, movie_frames, OTHER_TYPE);
    for (i = 0; i < movie_frames_max; POSINC(i))
    {
      camera[i].eye = NULL;
      camera[i].look = NULL;
      camera[i].up = NULL;
      camera[i].view_angle_x = -1.0;
      camera[i].view_angle_y = -1.0;
    }
    i = 0;
    movie_frames = 0;
    if (movie_eye != NULL)
    {
      while (NOT END_OF_LINE(movie_eye))
      {
        if (i == movie_frames_max)
          runtime_abort("too many MOVIE FRAMES (EYE)");
        ALLOCATE(camera[i].eye, xyz_struct, 1, OTHER_TYPE);
        get_valid(movie_eye, &(camera[i].eye->x), X_MIN, X_MAX,
                  "CAMERA EYE POINT X");
        get_valid(movie_eye, &(camera[i].eye->y), Y_MIN, Y_MAX,
                  "CAMERA EYE POINT Y");
        get_valid(movie_eye, &(camera[i].eye->z), Z_MIN, Z_MAX,
                  "CAMERA EYE POINT Z");
        ADVANCE(movie_eye);
        POSINC(i);
      }
      CLOSE(movie_eye);
      movie_frames = i;
      i = 0;
    }
    if (movie_look != NULL)
    {
      while (NOT END_OF_LINE(movie_look))
      {
        if (i == movie_frames_max)
          runtime_abort("too many MOVIE FRAMES (LOOK)");
        ALLOCATE(camera[i].look, xyz_struct, 1, OTHER_TYPE);
        get_valid(movie_look, &(camera[i].look->x), X_MIN, X_MAX,
                  "CAMERA LOOK POINT X");
        get_valid(movie_look, &(camera[i].look->y), Y_MIN, Y_MAX,
                  "CAMERA LOOK POINT Y");
        get_valid(movie_look, &(camera[i].look->z), Z_MIN, Z_MAX,
                  "CAMERA LOOK POINT Z");
        ADVANCE(movie_look);
        POSINC(i);
      }
      CLOSE(movie_look);
      if (i > movie_frames)
        movie_frames = i;
      i = 0;
    }
    if (movie_up != NULL)
    {
      while (NOT END_OF_LINE(movie_up))
      {
        if (i == movie_frames_max)
          runtime_abort("too many MOVIE FRAMES (UP)");
        ALLOCATE(camera[i].up, xyz_struct, 1, OTHER_TYPE);
        get_valid(movie_up, &(camera[i].up->x), X_MIN, X_MAX,
                  "CAMERA UP VECTOR X");
        get_valid(movie_up, &(camera[i].up->y), Y_MIN, Y_MAX,
                  "CAMERA UP VECTOR Y");
        get_valid(movie_up, &(camera[i].up->z), Z_MIN, Z_MAX,
                  "CAMERA UP VECTOR Z");
        ADVANCE(movie_up);
        POSINC(i);
      }
      CLOSE(movie_up);
      if (i > movie_frames)
        movie_frames = i;
      i = 0;
    }
    if (movie_angle != NULL)
    {
      while (NOT END_OF_LINE(movie_angle))
      {
        if (i == movie_frames_max)
          runtime_abort("too many MOVIE FRAMES (ANGLE)");
        get_valid(movie_angle, &(camera[i].view_angle_x), 0.5, 89.5,
                  "HORIZONTAL VIEW Angle");
        camera[i].view_angle_x = DEGREE_TO_RADIAN(camera[i].view_angle_x);
        get_valid(movie_angle, &(camera[i].view_angle_y), 0.5, 89.5,
                  "VERTICAL VIEW Angle");
        camera[i].view_angle_y = DEGREE_TO_RADIAN(camera[i].view_angle_y);
        ADVANCE(movie_angle);
        POSINC(i);
      }
      CLOSE(movie_angle);
      if (i > movie_frames)
        movie_frames = i;
      i = 0;
    }
  }

  if (verbose_mode == 0)
    return;

  options();
}
