/****************************************************************************
*
*						MegaGraph Graphics Library
*
*                   Copyright (C) 1993 Kendall Bennett.
*							All rights reserved.
*
* Filename:		$RCSfile: modes.c $
* Version:		$Revision: 1.2 $
*
* Language:		ANSI C
* Environment:	IBM PC (MS DOS)
*
* Description:	Routine to determine the available video modes for a
*				specified video card, given memory restrictions.
*
* $Id: modes.c 1.2 1993/03/07 04:04:13 kjb Exp $
*
* Revision History:
* -----------------
*
* $Log: modes.c $
* Revision 1.2  1993/03/07  04:04:13  kjb
* Bug fixes.
*
* Revision 1.1  1993/03/03  10:46:00  kjb
* Initial revision
*
****************************************************************************/

#include "drivers.h"

/*--------------------------- Global Variables ----------------------------*/

int		__modeTable[grMAXMODE+1];	/* Internal table of video modes	*/

/*------------------------- Implementation --------------------------------*/

/* Routines in 'detectgr.asm' for returning the list of possible video
 * modes for this adapter and display pages.
 */

char far * _getSuperVGAModes(void);
int _getSuperVGAPages(int mode,int memory,long *pagesize);

PRIVATE int loadVGAModes(int i)
/****************************************************************************
*
* Function:		loadVGAModes
* Parameters:	i	- Index to start loading modes at
* Returns:		Index of next available space in mode table.
*
* Description:	Fills in all of the standard VGA video modes in the mode
*				table.
*
****************************************************************************/
{
	__modeTable[i++] = grEGA_320x200x16;
	__modeTable[i++] = grEGA_640x200x16;
	__modeTable[i++] = grEGA_640x350x16;
	__modeTable[i++] = grVGA_640x400x16;
	__modeTable[i++] = grVGA_640x480x16;
	__modeTable[i++] = grVGA_320x200x256;
	__modeTable[i++] = grVGAX_320x200x256;
	__modeTable[i++] = grVGAX_320x240x256;
	__modeTable[i++] = grVGAX_320x400x256;
	__modeTable[i++] = grVGAX_320x480x256;
	__modeTable[i++] = grVGAX_360x200x256;
	__modeTable[i++] = grVGAX_360x240x256;
	__modeTable[i++] = grVGAX_360x400x256;
	__modeTable[i++] = grVGAX_360x480x256;
	return i;
}

PUBLIC int * MGL_availableModes(int driver,int memory)
/****************************************************************************
*
* Function:		MGL_availableModes
* Parameters:	driver	- Graphics device driver to check.
*				memory	- Amount of memory on video card
*
* Returns:		Returns a pointer to a list of all the video modes that are
*				currently available. This routine can be called before
*				MGL_init() is called to determine the video modes available.
*				The list of video modes is terminated by a -1, but you
*				will have to call MGL_detectGraph() to determine the
*				currently installed video card.
*
****************************************************************************/
{
	int		i;
	long	pagesize;
	char	far *modes;

	switch(driver) {
		case grMCGA:
			__modeTable[0] = grVGA_320x200x256;
			__modeTable[1] = -1;		/* Terminate the table	*/
			memory = 64;
			break;
		case grEGA:
			__modeTable[0] = grEGA_320x200x16;
			__modeTable[1] = grEGA_640x200x16;
			__modeTable[2] = grEGA_640x350x16;
			__modeTable[3] = -1;		/* Terminate the table	*/
			memory = 256;
			break;
		case grVGA:
			i = loadVGAModes(0);
			__modeTable[i] = -1;		/* Terminate the table	*/
			memory = 256;
			break;
		default:
			if (driver >= __FIRST_SVGA && driver <= __LAST_SVGA) {
				/* We have a Super VGA video card. We know that we can use
				 * all of the standard VGA video modes, but we must also
				 * determine what SuperVGA video modes are available. We
				 * obtain this information from the DetectGraph module.
				 */

				i = loadVGAModes(0);

				/* Install all detected SuperVGA modes, filtering
				 * those not supported with the memory installed
				 */

				modes = _getSuperVGAModes();
				if (memory >= 4096)	/* Fix strange memory sizes	*/
					memory = 4096;
				else if (memory >= 3072)
					memory = 3072;
				else if (memory >= 2048)
					memory = 2048;
				else if (memory >= 1024)
					memory = 1024;
				else if (memory >= 512)
					memory = 512;
				else if (memory >= 256)
					memory = 256;

				while (*modes != -1) {
					/* Compute the size of the video page for the mode, and
					 * if we have enough memory for one page then include
					 * it in the mode list
					 */

					if (*modes != 0) {
						if (_getSuperVGAPages(*modes | 0x80,memory,&pagesize) > 0) {
							if ((pagesize / 1024) <= memory)
								__modeTable[i++] = *modes;
							}
						}
					modes++;
					}
				__modeTable[i] = -1;		/* Terminate the table	*/
				}
			else {
				/* No video modes are supported for this adapter */

				__modeTable[0] = -1;		/* Terminate the table	*/
				}
		}
	return __modeTable;
}

PUBLIC int MGL_availablePages(int driver,int memory,int mode)
/****************************************************************************
*
* Function:		MGL_availablePages
* Parameters:	driver	- Graphics device driver to check.
*				memory	- Amount of memory on video card.
*				mode	- Video mode to check
* Returns:		Number of available video pages for the mode, -1 on error.
*
* Description:	Determines the number of available video pages for a
*				specific video mode and driver. This routine assumes that
*				the MGL_detectGraph() routine has already been called
*				to obtain the video device driver ID.
*
*				If the video mode specified is invalid, a -1 is returned.
*
****************************************************************************/
{
	int		*modes;
	long	pagesize;

	/* Get list of available video modes and check that the mode specified
	 * is currently valid
	 */

	modes = MGL_availableModes(driver,memory);

	while (*modes != -1) {
		if (mode == *modes)
			break;
		modes++;
		}
	if (*modes == -1)
		return -1;

	switch(driver) {
		case grMCGA:
			return 1;
		case grEGA:
		case grVGA:
			switch (mode) {
				case grEGA_320x200x16:
					return 8;
				case grEGA_640x200x16:
					return 4;
				case grEGA_640x350x16:
					return 2;
				case grVGA_640x400x16:
					return 2;
				case grVGA_640x480x16:
					return 1;
				case grVGA_320x200x256:
					return 1;
				}
			return -1;
		default:
			if (driver >= __FIRST_SVGA && driver <= __LAST_SVGA) {
				/* We have a Super VGA video card. We simply call the
				 * assembly language routine to compute the number
				 * of video pages.
				 */

				return _getSuperVGAPages(mode,memory,&pagesize);
				}
			else {
				/* No video modes are supported for this adapter */

				return -1;
				}
		}
}
