/* kfactor.c
 * Calculate stability factor K. Mainly used to analyse the results
 * of the famous PUFF program, as this is missing stability calculations.
 *
 * Stability can be expressed in terms of S-parameters.
 * The concept of stability can be expressed as:
 *
 *                   2        2    2
 *          1 - |S11|  - |S22|  + D
 *      K = ---------------------------
 *              2 * |S12 * S21|
 *
 *      where D = |S11 * S22 - S12 * S21|
 *
 * Definition: A linear 2-port is unconditional stable if K > 1 and D < 1.
 *
 * 7-Aug-1992
 * Peter Beyer  -PA3AEF-
 * beyer@athena.uto.dec.com
 */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

/* Definitions */
#ifndef __TURBOC__
struct complex {
    double x;                       /* Real part */
    double y;                       /* Imaginairy part */
};
#endif

typedef struct _PUFF {
        float f;
	struct complex *sp[4];      /* Order here must be S11 S22 S12 S21 !!!*/
        float K;
} PUFF;

/* Prototyping */
int stable(PUFF *p);

/* Operating system entry point */
main(argc, argv)
int argc;
char **argv;
{
    FILE *fp;
    char file[50];
    char line[150];                             /* Line to read in */
    struct complex spars[4] = {0, NULL, NULL, NULL, NULL, 0, 0};
    PUFF p;
    int i, j;
    char *sp[] = {{"S11"}, {"S22"}, {"S12"}, {"S21"}};
    int nspars = 0;
    char tmp[5][4];

    /* Check command line arguments */
    if (argc < 2) {
	printf("kfactor - calculate stability from PUFF output (c) PA3AEF\n");
	printf("Enter PUFF output file: ");
	gets(file);
    } else {
	strcpy(file, argv[1]);
    }

    /* Open .PUF file */
    if (!strstr(strupr(file), ".PUF"))
        strcat(file, ".PUF");
    if ((fp = fopen(file, "r")) == NULL) {
        printf("\nError opening %s\n",file);
        exit(0);
    }

    /* Find begin of S-parameter part in .PUF file */
    while (fgets(line, 150, fp) != NULL) {
	if ((strstr(line, "s{parameters}")) != NULL)
            break;
    }

    /* First line shows us the sequence of S-parameters */
    fgets(line, 150, fp);
    sscanf(&line, "%s %s %s %s %s", tmp[0], tmp[1], tmp[2], tmp[3], tmp[4]);
    /* Get the sequence right for our stability routine */
    for (i=0; i<4; i++) {
        for (j=0; j<4; j++) {
	    if ((strcmp(sp[j], tmp[i+1])) == 0) {
                p.sp[i] = &spars[j];
                nspars++;
                break;
            }
        }
    }

    /* Check mis-usage */
    if (nspars != 4) {
        printf("%s is missing! Need S11 S22 S12 S21.\n", sp[nspars]);
        exit(0);
    }

    /* Now for each line calculate what we want */
    while (fgets(line, 150, fp) != NULL) {
	if ((strstr(line, "{")) !=NULL)
	    break;
	sscanf(&line,"%f %le %le %le %le %le %le %le %le",
	    &p.f,
	    &(spars[0].x), &(spars[0].y),
	    &(spars[1].x), &(spars[1].y),
	    &(spars[2].x), &(spars[2].y),
	    &(spars[3].x), &(spars[3].y));
	if (!(stable(&p)))
	    printf("%7.3f GHz\tK = %le dB\tNot stable\n",p.f, p.K);
	else
	    printf("%7.3f GHz\tK = %le dB\tStable\n",p.f, p.K);
    }
}

/* Calculate Rollet's K-factor from S-parameters */
int stable(p)
PUFF *p;
{
    double D;
    struct complex t1, t2, t3;

    /* Calculate D */

    D = (p->sp[0]->x * p->sp[1]->x) - (p->sp[2]->x * p->sp[3]->x);

    /* Full K calculation */
    p->K = (1 +pow(D, 2) - pow(p->sp[0]->x, 2) - pow(p->sp[1]->x, 2)) /
	    (2 * p->sp[2]->x * p->sp[3]->x);

    /* Are we stable ? */
    if ((p->K < 1) || (D > 1))
        return(0);
    return(1);
}
