/******************************************************************************
* Iteraction library - pull down menus handler.				      *
*									      *
*					Written by Gershon Elber,  Oct. 1990  *
*******************************************************************************
* History:								      *
*  3 Oct 90 - Version 1.0 by Gershon Elber.				      *
******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdio.h>
#include "intr_loc.h"
#include "intr_gr.h"

#define PULL_DOWN_BORDER 10
#define PULL_DOWN_BASE_LINE 12

static int GRLastColor = 0;		       /* GR Color set before query. */

static void MenuProlog(void);
static void MenuEpilog(void);

/****************************************************************************
* Save current graphic state.						    *
****************************************************************************/
static void MenuProlog()
{
    /* Save current graphic state. */
    GRPushViewPort();
    _GRSetViewPort(0, 0, GRScreenMaxX, GRScreenMaxY);

    GRPushTextSetting();
    GRSetTextJustify(GR_TEXT_HJUSTIFY_LEFT, GR_TEXT_VJUSTIFY_BOTTOM);
    GRSetSTextStyle(GR_FONT_DEFAULT, GR_HORIZ_DIR, GR_TEXT_MAG_1);

    GRLastColor = GRGetColor();
}

/****************************************************************************
* Restore current graphic state.					    *
****************************************************************************/
static void MenuEpilog(void)
{
    GRPopTextSetting();
    GRSetColor(GRLastColor);

    GRPopViewPort();
}

/******************************************************************************
* Routine to free a pull down menu.					      *
******************************************************************************/
void IntrPullDownMenuDelete(IntrPullDownMenuStruct *PDMenu)
{
    _IntrFree(PDMenu);
}

/******************************************************************************
* Routine to create a pop up menu.					      *
* If SizeOfEntry = 0 then StrEntries is an array of (char *). Otherwise it    *
* holds the size of each entry in StrEntries which is a long string itself.   *
* NumOfEntries defines the length of the menu.				      *
* It should be noted that in both cases StrEntries is NOT copied.	      *
* ActionFuncs holds pointer to the action functions to be called upon menu    *
* selection. Functions are given no parameter and should return none.	      *
* It should be noted that ActionFuncs array is not copied.		      *
******************************************************************************/
IntrPullDownMenuStruct *IntrPullDownMenuCreate(char **StrEntries,
					       int SizeOfEntry,
                                               int NumOfEntries,
					       IntrIntFunc *ActionFuncs,
				               IntrColorType FrameColor,
				               IntrColorType BackColor,
				               IntrColorType ForeColor,
				               IntrColorType XorColor,
			                       int FrameWidth)
{
    IntrPullDownMenuStruct *PDMenu;

    PDMenu = (IntrPullDownMenuStruct *)
				_IntrMalloc(sizeof(IntrPullDownMenuStruct));
    PDMenu -> ForeColor = ForeColor;
    PDMenu -> BackColor = BackColor;
    PDMenu -> FrameColor = FrameColor;
    PDMenu -> XorColor = XorColor;
    PDMenu -> FrameWidth = FrameWidth;
    PDMenu -> NumOfEntries = NumOfEntries;
    PDMenu -> SizeOfEntry = SizeOfEntry;
    PDMenu -> StrEntries = StrEntries;
    PDMenu -> ActionFuncs = ActionFuncs;
    PDMenu -> _ActiveIndex = -1;

    return PDMenu;
}

/****************************************************************************
* Routine to change pop up menu entry i. Only menus defined using array of  *
* strings can be modified in this way (PDMenu -> SizeOfEntry must be zero). *
****************************************************************************/
void IntrPullDownSetEntry(IntrPullDownMenuStruct *PDMenu, char *Entry,
								int Index)
{
    if (PDMenu -> SizeOfEntry == 0)
        PDMenu -> StrEntries[Index] = Entry;
    else
        strncpy((char *) &PDMenu -> StrEntries[Index * PDMenu -> SizeOfEntry],
	        Entry,
		PDMenu -> SizeOfEntry - 1);
}

/****************************************************************************
* Routine to change pop up menu entry i. Only menus defined using array of  *
* strings can be modified in this way (PDMenu -> SizeOfEntry must be zero). *
****************************************************************************/
void IntrPullDownSetAction(IntrPullDownMenuStruct *PDMenu,
				IntrIntFunc ActionFunc, int Index)
{
    PDMenu -> ActionFuncs[Index] = ActionFunc;
}

/****************************************************************************
* Routine to change pop up menu frame width.				    *
****************************************************************************/
void IntrPullDownSetFrameWidth(IntrPullDownMenuStruct *PDMenu, int FrameWidth)
{
    PDMenu -> FrameWidth = FrameWidth;
}

/****************************************************************************
* Routine to change pop up menu colors.					    *
****************************************************************************/
void IntrPullDownSetColors(IntrPullDownMenuStruct *PDMenu,
		           IntrColorType FrameColor,
		           IntrColorType BackColor,
		           IntrColorType ForeColor,
		           IntrColorType XorColor)
{
    PDMenu -> ForeColor = ForeColor;
    PDMenu -> BackColor = BackColor;
    PDMenu -> FrameColor = FrameColor;
    PDMenu -> XorColor = XorColor;
}

/****************************************************************************
* Routine to draw the given pull down menu items at the given position.	    *
* The window is assumed to be fully visible.				    *
****************************************************************************/
void _IntrPullDownMenuDrawItems(IntrPullDownMenuStruct *PDMenu)
{
    int i;
    char
        **Str = PDMenu -> StrEntries;
    _IntrWindowStruct
	*Window = _IntrFindWndwUsingID(PDMenu -> WindowID);
    int Xmin = Window -> BBox.Xmin - Window -> FrameWidth,
	Xmax = Window -> BBox.Xmax + Window -> FrameWidth,
	Ymin = Window -> BBox.Ymin - Window -> FrameWidth - 1,
        Dx = (Xmax - Xmin) / PDMenu -> NumOfEntries,
        n = PDMenu -> NumOfEntries - 1;

    MenuProlog();

    for (i = 0; i <= n; i++) {
	_IntrWndwPutNameHeader(Xmin, i != n ? Xmin + Dx - 1 : Xmax, Ymin,
                               PDMenu -> FrameWidth,
                               PDMenu -> SizeOfEntry == 0 ?
				   Str[i] :
				   (char *) &Str[i * PDMenu -> SizeOfEntry],
			       FALSE, PDMenu -> FrameColor,
                               PDMenu -> ForeColor, PDMenu -> BackColor,
                               TRUE);
        Xmin += Dx;
    }
    PDMenu -> DrawnEntryWidth = Dx;

    if (PDMenu -> _ActiveIndex >= 0)
	_IntrPullDownInvertEntry(PDMenu -> _ActiveIndex, Window);

    MenuEpilog();
}

/****************************************************************************
* Routine to invert a given entry in a pull down menu.			    *
* It is assumed toe menu is whole visible.				    *
****************************************************************************/
void _IntrPullDownInvertEntry(int Index,
			      _IntrWindowStruct *Window)
{
    IntrPullDownMenuStruct
        *PDMenu = Window -> PDMenu;
    int Xmin = Window -> BBox.Xmin - Window -> FrameWidth,
	Xmax = Window -> BBox.Xmax + Window -> FrameWidth,
	Ymin = Window -> BBox.Ymin,
	Ymax = Window -> BBox.Ymin - (PDMenu -> FrameWidth << 1) - 1,
        Dx = (Xmax - Xmin) / PDMenu -> NumOfEntries;


    GRPushTextSetting();
    GRSetSTextStyle(GR_FONT_DEFAULT, GR_HORIZ_DIR, GR_TEXT_MAG_1);
    Ymin -= IntrWndwGetHeaderHeight("M", PDMenu -> FrameWidth) + 1;
    GRPopTextSetting();

    Xmin += Dx * Index;
    if (Index == PDMenu -> NumOfEntries - 1)
        Xmax -= PDMenu -> FrameWidth;
    else
        Xmax = Xmin + Dx - PDMenu -> FrameWidth - 1;
    Xmin += PDMenu -> FrameWidth;

    MenuProlog();

    IntrAllocColor(PDMenu -> XorColor, INTR_INTENSITY_VHIGH);
    GRXORRectangle(Xmin, Ymin, Xmax, Ymax);

    MenuEpilog();
}

/****************************************************************************
* Routine to attempt to match given location with one of the elements in    *
* items displayed. Item are displayed from Left to Right.		    *
* Returns the index that matches, or -1 if No match.			    *
****************************************************************************/
int _IntrPullDownMatchPosition(int x, int y,
			       _IntrWindowStruct *Window)
{
    IntrPullDownMenuStruct
        *PDMenu = Window -> PDMenu;
    int Xmin = Window -> BBox.Xmin,
	Xmax = Window -> BBox.Xmax,
	Ymin = Window -> BBox.Ymin + PDMenu -> FrameWidth -
        	IntrWndwGetHeaderHeight("M", PDMenu -> FrameWidth) - 1,
	Ymax = Window -> BBox.Ymin - PDMenu -> FrameWidth,
        Dx = (Xmax - Xmin) / PDMenu -> NumOfEntries;

    /* Fast clipping if cursor is not in the menu at all. */
    if (y > Ymax || y < Ymin) return -1;

    return (x - Xmin) / Dx;
}
