/* CAPTURE.H by Bruce Eckel Revolution2 Real-Time Consulting.
Preprocessor macros & function declarations for the RTD AD1000 board.
Note that for this code to work, the end-of-conversion
(EOC) line of the A/D converter must be strapped to
the parallel port B line 7 (PB7).  This is accomplished
by soldering a short wire from the EOC line to
the PB7 line on connector P8 on the AD1000 board (this
connector is clearly marked by silkscreen information on
the board).  This wire allows the code to monitor the
status of the A/D converter by looking at PB7.  */

#include <dos.h>  /* So you get Turbo C inport and outport macros
instead of functions.  The macros will execute faster. */

/* The macros in this header file generate no code unless
they are called (thus you can put them in the header file
without causing multiple-definition errors at link time).
Since the compiler does all the work of expanding the macro
at compile-time, you get the readability and ease-of-use
of function calls, but the run-time speed of in-line code.
Because of the code generated by the compiler for a function
call, small macros don't necessarily create more code than
the equivalent function call.
  Notice the macros don't have semicolons after them; this
forces the user to add the semicolon so the macro call has
the same syntax as a function call (for consistency).  */

/* The following address is set by a bird jumper clip on
the AD1000 board.  If you change the clip, simply change
the following (hexadecimal) number: */
#define base 0x200
/* Note that the compiler performs the addition of this number
in expressions like: base + 0xd
so no run-time overhead is incurred. */

/* The following preprocessor macro sets the analog
multiplexer address, so you can select which input the
A/D will read: */
#define MUX(channel) outportb(base + channel, 0)
/* (note that channels must be counted 0-7) */
/* The data written to the address is unimportant */

/* The following preprocessor macro starts a 12-bit
A/D conversion.  Again, the data written to the address
is unimportant: */
#define START_AD()  outport(base + 9, 0)

/* The following preprocessor macro tests PB7 to see
if the A/D conversion is completed.  A bitwise AND with
0x80 (high bit set) masks off the other bits: */
#define COMPLETE() (inportb(base + 0xd) & 0x80)

/* The following preprocessor macro produces the value
of the A/D conversion.  The backslash at the end of the
line allows you to continue the macro on the next line. */
#define READ_AD(result) { result = ( inportb(base + 8) << 4 ); \
	result += (inportb(base + 9) >> 4); }
/* Note the result is left-justified in the 16 bits of
   the A/D registers. */

/* The following preprocessor macro uses the previous macros
to start a reading, wait until the A/D completes the conversion,
and read the result: */
#define GET_VALUE(result) { \
  START_AD(); \
  while(!COMPLETE()) \
    ; \
  READ_AD(result); }

/* Here are the declarations for the functions which are
defined in CAPTURE.C.  By including this header file in
another code file, you automatically declare these
functions.  Thus, you can use the functions by including
the header and linking CAPTURE.OBJ at link time (see the
MAKEFILE for details). */
/* Display 16 bits of ones and zeroes: */
void print_binary(unsigned int);

/* Display value as a voltage between -5 and +5: */
void print_value(unsigned int reading);

/* The following function waits for the A/D input to
exceed "threshold" and then fills "buf" with "bufsize"
samples, taken at software-controlled sampling rate "rate."
Notice that the effect of "rate" will vary depending on the
type and speed of your machine: */
void capture(unsigned * buf, int bufsize, unsigned threshold, int rate);
