/*
 * WMF Import Filter - UTILITY FUNCTIONS
 *
 * LANGUAGE      : Microsoft C6.0
 * MODEL         : medium
 * ENVIRONMENT   : Microsoft Windows 3.0 SDK
 * STATUS        : operational
 *
 * This module contains utility functions used throughout the
 * WMF import filter.
 *
 *    Eikon Systems, Inc.
 *    989 East Hillsdale Blvd, Suite 260
 *    Foster City, California 94404
 *    415-349-4664
 *
 * 08/01/91 1.00 - David E. West - initial creation.
 * 09/01/91 1.01 - Kevin P. Welch - minor revisions.
 *
 */

#define NOCOMM

#include <windows.h>
#include <limits.h>
#include <dos.h>
#include "wmf.h"

/* local definitions */
#define MAX_READ		0x10000L

/* local function definitions */
static WORD NEAR PASCAL		MyRead( INT, LPSTR, DWORD, LPDWORD );

/**/

/*
 * hread( hfSrc, hpb, cbToRead ) : DWORD;
 *
 *    hfSrc          handle to file to read from
 *    hpb            handle to buffer to read into
 *    cbToRead       number of bytes to read
 *
 * This function reads cbRead bytes from hfSrc into the memory block hpb.
 * The buffer can start anywhere in memory and cross 64K boundaries.  The
 * amount read can be more than 64K.  This function will do multiple reads
 * if necessary.
 *
 * The return value is the number of bytes read.
 *
 */

DWORD FAR PASCAL hread(
	INT            hfSrc,
	HPSTR          hpb,
	DWORD          cbToRead )
{
	BOOL				fContinue;
	WORD				idErr;
	DWORD				cbRead;
	DWORD				cbSize;
	DWORD				cbTotal;

	/* initialize */
	cbTotal = 0;
	cbSize = MAX_READ - FP_OFF(hpb);		/* up to segment boundary */

	/* read bytes */
	fContinue = TRUE;
	while ((cbToRead >= cbSize) && fContinue)
	{
		idErr = MyRead( hfSrc, hpb, cbSize, &cbRead );
		if (idErr == 0)
		{
			cbToRead -= cbRead;
			hpb += cbRead;
			cbTotal += cbRead;
			cbSize = MAX_READ - FP_OFF(hpb);		/* up to segment boundary */
		}
		else
			fContinue = FALSE;
	}

	/* read last segment */
	if (fContinue && (cbToRead > 0))
	{
		idErr = MyRead( hfSrc, hpb, cbToRead, &cbRead );
		if (idErr == 0)
			cbTotal += cbRead;
	}

	return cbTotal;
}

/**/

/*
 * MyRead( hf, lpb, cbToRead, lpcbRead ) : WORD;
 *
 *    hf             handle to file to read
 *    lpb            pointer to buffer to read into
 *    cbToRead       number of bytes to read (<=64K)
 *    lpcbRead       number of bytes actually read
 *
 * This function reads up to 64K bytes from the file hf into the buffer
 * lpb.  The offset of lpb must be <= the number of bytes to read, since
 * the DOS read function will not cross segment boundaries in the buffer
 * in which it places data.  This error condition is not checked for!!!
 *
 * The number of bytes actually read is returned in *lpcbRead.
 *
 * The return value is zero if the function is successful, and a nonzero
 * error code otherwise.
 *
 */

static WORD NEAR PASCAL MyRead(
	INT            hf,
	LPSTR          lpb,
	DWORD          cbToRead,
	LPDWORD			lpcbRead )
{
	WORD				idErr;
	DWORD				cbTotal;
	DWORD				cbMaxRead;

	/* initialize */
	idErr = 0;
	cbTotal = 0;

	cbMaxRead = 64L * 1024L;
	if ((cbToRead > 0) && (cbToRead <= cbMaxRead))
	{
		WORD			cbRead;
		WORD			cbFirst;
		WORD			cbSecond;

		cbFirst = (cbToRead < (0xffff-1)) ? (WORD)cbToRead : (WORD)(cbToRead/2);	
		cbSecond = (WORD)(cbToRead - cbFirst);

		/* read first block */
		cbRead = _lread( hf, lpb, cbFirst );
		if (cbRead != -1)
		{
			/* read second block */
			cbTotal += cbRead;
			lpb += cbRead;
			if (cbSecond > 0)
			{
				cbRead = _lread( hf, lpb, cbSecond );
				if (cbRead != -1)
					cbTotal += cbRead;
				else
					idErr = 1;
			}
		}
		else
			idErr = 1;
	}
	else
		idErr = 1;

	/* assign number of bytes read */
	if (idErr == 0)
		*lpcbRead = cbTotal;

	return idErr;
}
