/***************************************************************************
 *  NAMESRV.C								   *
 *	This file contains following NetBIOS Name Service related functions*
 *									   *
 *	  AddName  () - Adds a unique name or group name in the Name Table *
 *	  DelName  () - Deletes a name from the Name Table		   *
 *	  FindName () - Find a name on the networks			   *
 *									   *
 *  History:	Alok Sinha  October, 1991    Created			   *
 *									   *
 ***********************************************************************/

// Include files
#include <ncb.h>
#include <common.h>
#include <namesrv.h>
#include <memory.h>
#include <stdio.h>
#include <string.h>

/*
 * FUNCTION    :: AddName()
 * PARAMETERS  ::
 *	 [in]  pchName	  :- 16 byte unique or global name pointer
 *	 [in]  ucFlags	  :- such as Unique Name or Global Name
 *	 [in]  ucLana	  :- Lan Adaptor number
 *	 [out] pucNameNum :- Pointer to Name Number. A valid value can be
 *			     expected in this, if return value is NO_ERROR.
 * RETURN VALUE:
 *	    NO_ERROR if no errors were encountered, else error returned from
 *	    NetBIOS call in Retcode field.
 */


unsigned char AddName (char *pchName,
			unsigned char ucFlags,
			unsigned char ucLana,
			unsigned char * pucNameNum
		       )
{
    NCB Ncb;
    unsigned char ucRc;

    /* First clear out the buffer */
    ClearNcb ( &Ncb);

    /*	Set the Command 	 */
    if ( ucFlags == ADD_UNIQUE_NAME)
	Ncb.ncb_command = NCBADDNAME;
    else if ( ucFlags == ADD_GROUP_NAME)
	Ncb.ncb_command = NCBADDGRNAME;
    else
	return ERROR_INVALID_PARAMETER;

    /* set the LAN Adaptor number */
    Ncb.ncb_lana_num = ucLana;

    /* set the name		  */
    memcpy ( Ncb.ncb_name, pchName,  NCBNAMSZ);

    /* Make the OS dependent NetBIOS call */
    if ((ucRc=NetBiosRequest ( &Ncb))!= NO_ERROR)
	return ucRc ;

    /* return name number if call was successful */
    if (Ncb.ncb_retcode == NO_ERROR)
	*pucNameNum = Ncb.ncb_num;

    return Ncb.ncb_retcode;
}


/*
 * FUNCTION    :: DelName()
 * PARAMETERS  ::
 *	 [in]  pchName	  :- 16 byte unique or global name pointer
 *	 [in]  ucLana	  :- Lan Adaptor number
 *	 [in]  ucNameNum :-  Name Number associated with Name.
 *
 * RETURN VALUE:
 *	    NO_ERROR if no errors were encountered, else error returned from
 *	    NetBIOS call in Retcode field.
 */


unsigned char DelName (char *pchName,
			unsigned char ucLana,
			unsigned char ucNameNum
		       )
{
    NCB Ncb;
    unsigned char ucRc;

    /* First clear out the buffer */
    ClearNcb ( &Ncb);

    /*	Set the Command 	 */
    Ncb.ncb_command = NCBDELNAME;

    /* set the LAN Adaptor number */
    Ncb.ncb_lana_num = ucLana;

    /* set the name		  */
    memcpy ( Ncb.ncb_name, pchName,  NCBNAMSZ);

    /* Make the OS dependent NetBIOS call   */
    if ((ucRc=NetBiosRequest ( &Ncb))!= NO_ERROR)
	return ucRc ;

    return Ncb.ncb_retcode;
}

/*
 * FUNCTION    :: FindName()
 * PARAMETERS  ::
 *	 [in]  pchName	  :- 16 byte unique or global name pointer
 *	 [in]  ucLana	  :- Lan Adaptor number
 *
 * RETURN VALUE:
 *	    TRUE  :- If a workstation with Supplied name exists and reponds.
 *	    FALSE :- False otherwise
 */


BOOL FindName (char *pchName,
		unsigned char ucLana
	      )
{
    NCB 	   Ncb;
    unsigned char ucRc;
    unsigned char  ucBuffer [ FIND_BUFFER_LENGTH ];
    FINDBUF	   *pFindBuf;

    /* First clear out the buffer */
    ClearNcb ( &Ncb);

    /*	Set the Command 	 */
    Ncb.ncb_command = NCBFINDNAME;

    /* set the LAN Adaptor number */
    Ncb.ncb_lana_num = ucLana;

    /* set the name		  */
    memcpy ( Ncb.ncb_callname, pchName,	NCBNAMSZ);

    /* set the buffer and buffer length */
    Ncb.ncb_buffer = (char far *) ucBuffer;
    Ncb.ncb_length =  FIND_BUFFER_LENGTH;

    /* Make the OS dependent NetBIOS call     */
    if ((ucRc=NetBiosRequest ( &Ncb))!= NO_ERROR)
	return ucRc ;

    /* check the length returned	*/
    if (Ncb.ncb_length < sizeof (FINDBUF) )
    {
	printf ("retcode: [%x] length: [%d]\n",Ncb.ncb_retcode, Ncb.ncb_length );
	return FALSE;
    }

    /* cast FINDBUF pointer into the buffer */
    pFindBuf = (FINDBUF *) ucBuffer;

    /*
     * if any one workstation with supplied name responded,
     * we return TRUE since the name supplied can be registered
     * as a unique name or group name.
     */

    if (pFindBuf->usNumResponding != 0)
    {
	printf("Responding: %d\n",pFindBuf->usNumResponding);
	printf("status: %d\n", pFindBuf->ucStatus);
	printf("hdr length: %d\n", pFindBuf->ucLanHeaderLength);
	return TRUE;
    }
    return FALSE;
}
