/* sfind.c
    Searches all Microsoft Word .DOC files in the default directory
        for a string.
    To compile MSC 4.0/5.0 and QuickC 1.0:  cl sfind.c
 */
    
#include <stdio.h>
#include <dos.h>
#include <string.h>

#define DYEAR(x) (x>>9)+80     /* These macros breakdown the calendar and */
#define DMON(x)  x>>5 & 0x000f /* time info to individual bytes */
#define DDAY(x) x & 0x001f
#define DHOUR(x) x>>11
#define DMIN(x) x>>5 & 0x003f
#define DSEC(x) x<<1 & 0x003f

#define BUFSIZE 30000             /* Program can hold a max of 30k bytes */
#define MAXFILES 500
#define WORD_HEADER_SIZE 128      /* size of Word's header in .DOC file */

struct fdir
{ 
    unsigned char fname[13];      /* string containing the filename */ 
    unsigned char attrib;         /* file attribute */
    int fhour;                    /* time */
    int fmin;
    int fsec;  
    int fday;                     /* date */
    int fmon;
    int fyear;                        
    long fsize;                   /* size of file */ 
} fdata[MAXFILES];                /* program can read up to MAXFILES files */

struct find_t ff_area;            /* defined in dos.h */

char *scompare(FILE *,char *,const char *,int);

main(int argc,char *argv[])
{
    int numfound=0;
    const char *string;         /* pointer to text to find */

    if(argc<2)
    {
        printf("Usage: SFIND <string to search>");
        exit(1);
    }
    string=strupr(argv[1]);
    
    if( (numfound = fsearch(string)) && numfound != -1)
        display_fileinfo(numfound,string);
    else
        printf("\n0 files found containing \"%s\"\n",string);
}

    /* fsearch()
        Allocates the search buffer.
        Finds the first .DOC file in the current directory.
        Skips past header of .DOC file.
        Calls scompare() to read the remainder of the file and search for
            the string.
        If found, puts the file information into a structure of the
            fdata array.
        Continues until all .DOC files are processed.
        Returns -1 on memory allocation error.
        Otherwise, returns the number of files found that contain the string.
    */
fsearch(const char *string)
{
    int flag;                      /*equals 0 while *.DOC files are found*/
    long loop;                     /*counts loops for files>30k*/
    int ct=0;                      /*keeps count of files found*/
    char *ffind='\0';              /*not equal to '\0' if string found*/
    FILE *ptr;                     /*pointer to FILE structure*/
    char *buffer;                  /*holds text from file*/

    if(!(buffer=(char *)malloc(BUFSIZE+1)))
    {
        printf("Memory allocation error");
        return(-1);
    }

        /* find the first file, then find all of the rest */
    for( flag=(_dos_findfirst("*.doc",_A_NORMAL,&ff_area));    !flag;
        flag = (_dos_findnext(&ff_area)))      
    {
        if(!(ptr=fopen(ff_area.name,"rb")))
        {
            printf("\nsfind: unable to open %.13s\n",ff_area.name);
            continue;
        }
        printf("SEARCHING %.13s\n",ff_area.name);
        
                                /* skip WORD_HEADER_SIZE bytes of WORD code */
        fseek(ptr,(long)WORD_HEADER_SIZE,SEEK_SET);        
    
                                /*if file too large for buffer */
        if((ff_area.size-WORD_HEADER_SIZE) > BUFSIZE)      
        {
            for(loop = ff_area.size-WORD_HEADER_SIZE; loop && !ffind;
                loop -= BUFSIZE)
                ffind=scompare(ptr,buffer,string,BUFSIZE);
        }
        else
            ffind=scompare(ptr,buffer,string,
                (int)(ff_area.size-WORD_HEADER_SIZE));
    
        if(ffind)                          /*if string found...*/
        {
            strncpy(fdata[ct].fname,ff_area.name,13); /*save file data*/
            fdata[ct].attrib=ff_area.attrib;
            fdata[ct].fhour=DHOUR(ff_area.wr_time);
            fdata[ct].fmin=DMIN(ff_area.wr_time);
            fdata[ct].fsec=DSEC(ff_area.wr_time);
            fdata[ct].fday=DDAY(ff_area.wr_date);
            fdata[ct].fmon=DMON(ff_area.wr_date);
            fdata[ct].fyear=DYEAR(ff_area.wr_date);
            fdata[ct].fsize=ff_area.size;
            ++ct;
            if(ct == MAXFILES)
                break;
        }        
        fclose(ptr);
        ffind='\0';                          /*reset to NULL*/
    }                                        /*end of FOR loop*/
    free(buffer);
    return(ct);
}

    /* display_fileinfo()
        Prints information on each file that contains the string.
     */
display_fileinfo(int numfound,char *string)
{
    int i;

    printf("\n%d file%s contain%s \"%s\":\n",
        numfound,(numfound == 1 ? "" : "s"),(numfound != 1 ? "" : "s"),
            string);
    for(i = 0; i < numfound; ++i)
        printf("%13s %7lu bytes  %02d-%02d-%02d  %02d:%02d:%02d\n",
            fdata[i].fname,
            fdata[i].fsize,
            fdata[i].fmon,
            fdata[i].fday,
            fdata[i].fyear,
            fdata[i].fhour,
            fdata[i].fmin,
            fdata[i].fsec);
}

    /* scompare()
        Reads the contents of the file pointed to by ptr. (Up to fsize bytes)
        Uppercases the buffer, to match any case.
        Searches for the string.
        Returns non-zero value if found.
        Otherwise, returns 0.
     */
char *scompare(FILE *ptr,char *buffer,const char *string,int fsize)
{
    char *ffind='\0';                  /*will not equal '\0' if string found*/
    unsigned int numread;              /*number of chars read from file*/
    const char *hold;                  /*pointer to file data to search*/
    
    numread=fread((char *)buffer,sizeof(char),fsize,ptr);
    
    if(numread)                       /*if not an empty file*/
    {
        buffer[numread] = NULL;
        hold = strupr(buffer);        /*capitalize all the characters*/
        ffind = strstr(hold,string);  /*search for string*/
    }
    return(ffind);
}

