/**************************************************************************
*   SERIAL.CPP - Listing 2
*   Written by Kevin D. Weeks, August 1990
*   Compiles and runs under Borland Turbo C++ and Zortech C++.
*/
#include <stdio.h>
#include "serial.hpp"
// initialize the tables for use by Int 14h. the enum's defined in
// SERIAL.HPP (Buad_Rate, Parity, etc.) index into these tables
unsigned int Serial_Comm::baud_table[8] = {0,0x20,0x40,0x60,
                                        0x80,0xa0,0xc0,0xe0};
unsigned int Serial_Comm::parity_table[3] = {8,0x18,0};
unsigned int Serial_Comm::stop_table[2] = {0,4};
unsigned int Serial_Comm::data_table[2] = {2,3};
Serial_Comm  *Serial_Comm::open_flag = NULL;
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Null constructor. Sets the data attributes to arbitrary
*   defaults. If some other values seem better, use them.           */
Serial_Comm::Serial_Comm(void)
{
    com_port = Com_1;
    baud_rate = Baud_1200;
    parity = No_Parity;
    stop_bits = Stop_Bits_1;
    data_bits = Data_Bits_8;
    buffer_size = DEFAULT_BUF_SIZE;
    recv_buffer = new unsigned char[buffer_size];
    send_buffer = new unsigned char[buffer_size];
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Constructor - allows the user to specify start-up parameters.   */
Serial_Comm::Serial_Comm(Com_Port port, Baud_Rate baud, Parity par,
                         Stop_Bits stop, Data_Bits data)
{
    unsigned int    parameters;
    com_port = port;
    baud_rate = baud;
    parity = par;
    stop_bits = stop;
    data_bits = data;
    buffer_size = DEFAULT_BUF_SIZE;
    recv_buffer = new unsigned char[buffer_size];
    send_buffer = new unsigned char[buffer_size];
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Destructor - closes the port if open and owned by this instance
*   and frees the buffers.                                          */
Serial_Comm::~Serial_Comm(void)
{
    close();
    delete [buffer_size] recv_buffer;
    delete [buffer_size] send_buffer;
    recv_buffer = send_buffer = NULL;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   First, set the local buffers for the assembly module. Then open
*   the port and claim it. Fails if someone else owns the port.     */
Result  Serial_Comm::open(void)
{
    unsigned int    parameters;
    if (open_flag) return ERROR;
    com_set_buffers(recv_buffer,send_buffer,buffer_size);
    parameters = baud_table[baud_rate] | parity_table[parity] |
                 stop_table[stop_bits] | data_table[data_bits];
    if (com_open(com_port,parameters) != ERROR) {
        open_flag = this;                   // claim the port
        return OK;
    }
    return ERROR;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   If the port is open and belongs to this instance, close it.     */
void    Serial_Comm::close(void)
{
    if (open_flag == this) {
        com_close();
        open_flag = NULL;
    }
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Read whatever's in the local buffer into the client's buffer.
*   Fail if someone else owns the port.                             */
int  Serial_Comm::read(void *client_buffer)
{
    if (open_flag == this) return com_read(client_buffer);
    else return -1;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Write the client's buffer. Fails if already sending or if
*   someone else owns the port.                                     */
Result   Serial_Comm::write(void *buffer, int num_bytes)
{
    if (open_flag == this) return (Result)com_write(num_bytes,buffer);
    else return ERROR;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Read a single character from the receive buffer.                */
int     Serial_Comm::read_char(void)
{
    if (open_flag == this) return com_read_char();
    else return -1;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Write a single character out the port. Note: this method will
*   fail if a message is currently being sent.                      */
Result   Serial_Comm::write_char(unsigned char chr)
{
    if (open_flag == this) return com_write_char(chr);
    else return ERROR;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Clear and re-initialize the recive buffer.                      */
Result  Serial_Comm::clr_recv_buffer(void)
{
    if (open_flag == this || open_flag == NULL) {
        com_clr_recv_buf();
        return OK;
    }
    return ERROR;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Clear and re-initialize the send buffer. If currently sending
*   the remainder of the message will be discarded and the trans-
*   mit interrupt disabled.                                         */
Result  Serial_Comm::clr_send_buffer(void)
{
    if (open_flag == this || open_flag == NULL) {
        com_clr_send_buf();
        return OK;
    }
    return ERROR;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Get the current port status.                                    */
Comm_Status  Serial_Comm::get_status(void)
{
    Comm_Status stat;
    if (open_flag == this || open_flag == NULL)
        stat.value = com_get_status();
    else stat.value = 0;
    return stat;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Returns the number of bytes in the receive buffer.              */
int  Serial_Comm::get_bytes_recvd(void)
{
    if (open_flag == this) return com_chars_recvd();
    else return -1;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Returns the number of bytes sent from the current message.      */
int  Serial_Comm::get_bytes_sent(void)
{
    if (open_flag == this) return com_chars_sent();
    else return -1;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Specifies the com port to use.                                  */
void    Serial_Comm::set_port(Com_Port port)
{
    com_port = port;
    if (open_flag == this) {
        close();
        open();
    }
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Specifies the baud rate.                                        */
void    Serial_Comm::set_baud_rate(Baud_Rate baud)
{
    baud_rate = baud;
    if (open_flag == this) {
        close();
        open();
    }
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Specifies the parity.                                           */
void    Serial_Comm::set_parity(Parity par)
{
    parity = par;
    if (open_flag == this) {
        close();
        open();
    }
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Specifies the number of stop bits.                              */
void    Serial_Comm::set_stop_bits(Stop_Bits stop)
{
    stop_bits = stop;
    if (open_flag == this) {
        close();
        open();
    }
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Specifies the number of data bits.                              */
void    Serial_Comm::set_data_bits(Data_Bits data)
{
    data_bits = data;
    if (open_flag == this) {
        close();
        open();
    }
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*   Specifies the size of low-level buffers.                        */
Result      Serial_Comm::set_buffer_size(int size)
{
    Bool    port_open = FALSE;
    // close the port if it's open and owned by this instance
    if (open_flag == this) {
        port_open = TRUE;
        close();
    }
    // free the original buffers, then change the buffer size and
    // re-allocate the buffers.
    delete [buffer_size] recv_buffer;
    delete [buffer_size] send_buffer;
    buffer_size = size;
    recv_buffer = new unsigned char[buffer_size];
    send_buffer = new unsigned char[buffer_size];
    // if the port was open re-open it and re-set the buffers
    if (port_open == TRUE) {
        com_set_buffers(recv_buffer,send_buffer,buffer_size);
        return open();
    }
    return OK;
}
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
unsigned int clock_ticks(int start)
{
    static long     reference;
    static long far *bios_clock = (long far *)0x0040006c;

    if (start)
        reference = *bios_clock;
    return (unsigned int)((*bios_clock - reference) / 2);
}
