// pardonme.cpp -- Windows alarm program by Tom Swan

// Notes and suggestions
// - hours must be in 24-hour format
// - alarm times must be unique
// - sort alarm times from low to high
// - store alarms in a file so they can be edited
// - abort program if no alarms are set
// - display running time in dialog
// - display list of alarms

#include <owl.h>   // ObjectWindows header
#include <time.h>  // Standard library header
#include "ids.h"   // Resource IDs header

#define APP_NAME "PardonMe"  // Application name
#define TIMER_ID 1           // Timer identifier
#define TIMER_DELAY 30000    // Half minute intervals
#define EM_RUNNING -100      // Program-already-running error code
#define EM_OUTOFTIMERS -101  // Out-of-timers error code
#define DEFAULTMSG "No alarms set"  // Default dialog message
#define NUMALARMS 5          // Number of alarms

struct Alarm {    // Alarm structure
  int alarmTime;         // Alarm time in minutes
  int alarmSet;          // True (nonzero) if alarm is set
  const char *alarmMsg;  // Pointer to dialog message
};

Alarm alarms[NUMALARMS];  // Global array of alarm structures

// The application class
class TPardonApp: public TApplication
{
public:
  TPardonApp(LPSTR aName, HANDLE hInstance, HANDLE hPrevInstance,
    LPSTR lpCmd, int nCmdShow): TApplication(aName, hInstance,
    hPrevInstance, lpCmd, nCmdShow) { };
  void Error(int ErrorCode);
  void InitInstance();
  virtual void InitMainWindow();
};

// The main window (as a modeless dialog!)
class TPardonDlg: public TDialog
{
protected:
  int timerSet;  // True (nonzero) if timer is set
  int theDay;    // Current day (full date not needed)
  int theTime;   // Current time in minutes
  const char *theMsg;  // Pointer to alarm message
public:
  TPardonDlg(PTWindowsObject AParent, int ResourceId)
    : TDialog(AParent, ResourceId) { }
// Inherited member functions
  virtual BOOL CanClose();
  virtual LPSTR GetClassName();
  virtual void GetWindowClass(WNDCLASS &AWndClass);
  void SetupWindow();
// New member functions
  void AssignAlarm(int n, int hour, int minute, const char *msg);
  void GetDayTime(int &d, int &t);
  void SetAlarms(void);
// Message response member functions
  virtual void IDAbout(RTMessage)
    = [ID_FIRST + ID_ABOUT];
  virtual void IDClose(RTMessage)
    = [ID_FIRST + ID_CLOSE];
  virtual void IDHide(RTMessage)
    = [ID_FIRST + ID_HIDE];
  virtual void WMDestroy(RTMessage)
    = [WM_FIRST + WM_DESTROY];
  virtual void WMSysCommand(RTMessage)
    = [WM_FIRST + WM_SYSCOMMAND];
  virtual void WMTimeChange(RTMessage)
    = [WM_FIRST + WM_TIMECHANGE];
  virtual void WMTimer(RTMessage)
    = [WM_FIRST + WM_TIMER];
};

// Display error message. "Falls through" to end program.
void TPardonApp::Error(int ErrorCode)
{
  LPSTR msg;            // Long pointer to string
  char string[40];      // Buffer for wsprintf()
  switch (ErrorCode) {
    case EM_OUTOFTIMERS:
      msg = "Out of timers";
      break;
    case EM_RUNNING:
      msg = "This program is already running";
      break;
    default:
      msg = "Window creation error";
  }
  wsprintf(string, "Error %d: %s", ErrorCode, msg);
  MessageBox(NULL, string, APP_NAME, MB_ICONSTOP | MB_OK);
}

// Initialize program instance (only one allowed)
void TPardonApp::InitInstance()
{
  if (hPrevInstance)
    Status = EM_RUNNING;  // Error: program already running
  else
    TApplication::InitInstance();
}

// Initialize the application's main window
void TPardonApp::InitMainWindow()
{
  MainWindow = new TPardonDlg(NULL, ID_DIALOG);
  nCmdShow = SW_SHOWMINNOACTIVE;
}

// Return true if okay to close dialog
BOOL TPardonDlg::CanClose()
{
  return (MessageBox(HWindow, "End program now?",
    APP_NAME, MB_YESNO) == IDYES);
}

// Return window class name
LPSTR TPardonDlg::GetClassName()
{
  return "TPardonDlg";  // Same as dialog resource class name!
}

// Return window class to be registered with Windows
void TPardonDlg::GetWindowClass(WNDCLASS &AWndClass)
{
  TDialog::GetWindowClass(AWndClass);
  AWndClass.hIcon = LoadIcon(GetApplication()->hInstance,
    MAKEINTRESOURCE(ID_ICON));
}

// Initialize window and set the timer
void TPardonDlg::SetupWindow()
{
  TDialog::SetupWindow();
  GetDayTime(theDay, theTime);
  theMsg = DEFAULTMSG;
  SetAlarms();
// - Set a timer to begin generating WM_TIMER messages
  timerSet = SetTimer(HWindow, TIMER_ID, TIMER_DELAY, NULL);
  if (!timerSet)
    Status = EM_OUTOFTIMERS;  // End program if timer not set
}

// Get current day number and time in minutes
void TPardonDlg::GetDayTime(int &d, int &t)
{
  time_t secs;
  struct tm *tmp;
  time(&secs);  // Get time in seconds from 1/1/1970
  tmp = localtime(&secs);  // Convert time to struct
  d = tmp->tm_mday;  // Set d to the day number
  t = tmp->tm_hour * 60 + tmp->tm_min;  // Calculate minutes
}

// Assign hour, minute, and msg to alarm number n
// Sets alarm to ON if time is >= current time
void TPardonDlg::AssignAlarm(int n, int hour, int minute,
  const char *msg)
{
  if (n < 0 || n >= NUMALARMS)
    return;  // Ignore index n range errors
  int t = hour * 60 + minute;  // Calculate time in minutes
  alarms[n].alarmTime = t;
  alarms[n].alarmSet = (t >= theTime);
  alarms[n].alarmMsg = msg;
}

// Set or reset all alarms (hours in 24-hour format)
void TPardonDlg::SetAlarms(void)
{
//            n  hr  mn  message
  AssignAlarm(0,  9, 00, "Good morning! Have a great day!");
  AssignAlarm(1, 11, 50, "Lunch time in 10 minutes. Hungry?");
  AssignAlarm(2, 15, 30, "Take a break. You deserve it!");
  AssignAlarm(3, 17, 15, "Backup your files.");
  AssignAlarm(4, 18, 10, "Are you still working? Get a life!");
}

// Display about-box dialog
void TPardonDlg::IDAbout(RTMessage)
{
  TDialog *dp = new TDialog(this, ID_ABOUTDLG);
  GetApplication()->ExecDialog(dp);
}

// Respond to selection of dialog's Close button
void TPardonDlg::IDClose(RTMessage)
{
  CloseWindow();
}

// Respond to selection of dialog's Hide button
void TPardonDlg::IDHide(RTMessage)
{
  Show(SW_MINIMIZE);  // Shrink window back to a pumpkin
}

// Respond to WMDestroy message. Kill timer
void TPardonDlg::WMDestroy(RTMessage msg)
{
  TDialog::WMDestroy(msg);
  if (timerSet)
    KillTimer(HWindow, TIMER_ID);
}

// Trap system-menu Close command
void TPardonDlg::WMSysCommand(RTMessage msg)
{
  if (msg.WParam == SC_CLOSE) // This step required to enable
    CloseWindow();            //  CanClose() function!
  else
    DefWndProc(msg);          // Windows handles other messages
}

// Reset all alarms if system time is changed
void TPardonDlg::WMTimeChange(RTMessage)
{
  SetAlarms();
}

// Respond to WMTimer messages every 30 seconds or so
void TPardonDlg::WMTimer(RTMessage)
{
  int testDay;
  GetDayTime(testDay, theTime);  // Get current day and time
  if (testDay != theDay) {       // If the day has changed
    theDay = testDay;            //   save new day number
    SetAlarms();                 //   reset all alarms
  }
  for (int i = 0; i < NUMALARMS; i++) {
    if (theTime >= alarms[i].alarmTime && alarms[i].alarmSet) {
      alarms[i].alarmSet = 0;      // Turn off the alarm
      theMsg = alarms[i].alarmMsg; // Assign message address
      SendDlgItemMsg(ID_TEXT, WM_SETTEXT, 0, long(theMsg));
      Show(SW_SHOWNORMAL);         // Open possibly iconic window
      BringWindowToTop(HWindow);   // Bring open window forward
      MessageBeep(0);              // Get user's attention
      return;                      // Exit function immediately
    }
  }
}

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance,
  LPSTR lpCmdLine, int nCmdShow)
{
  TPardonApp MiniApp(APP_NAME, hInstance, hPrevInstance,
    lpCmdLine, nCmdShow);
  MiniApp.Run();
  return MiniApp.Status;
}
