(* Turtle light *)

#open "graphics";;
#open "float";;

open_graph "";;

let radius = 10;;
let background = black;;
let foreground = white;;

let xcenter = ref 0;;
let ycenter = ref 0;;

type state = {
  mutable x : float;           (* Center of screen at 0.0 *)
  mutable y : float;           (* Idem. *)
  mutable heading : float;     (* Clockwise, 0.0 at 12 o'clock. *)
  mutable down : bool;         (* True if moving the turtle will draw. *)
  mutable color : color;
  mutable width : int;
  mutable shown : bool
};;

let t = { x = 0.0; y = 0.0; heading = 0.0;
          down = true; color = foreground; width = 1; shown = true
};;

let saved_background = create_image (mult_int radius 2) (mult_int radius 2);;

let round x =
  if x >= 0.0
  then int_of_float (x + 0.5)
  else minus_int (int_of_float (-x + 0.5))
;;

let cvx x = add_int (round x) !xcenter;;
let cvy y = add_int (round y) !ycenter;;
let pi180 = 2.0 * 3.14159265358979 / 360.0;;
let cvh h = (-h + 90.0) * pi180;;
let rec mod360 h =
  if h >= 0.0
  then h - 360.0 * float_of_int (int_of_float (h / 360.0))
  else 360.0 - mod360 (- h)
;;

let begin_drawing () =
  if t.shown
  then draw_image saved_background (sub_int (cvx t.x) radius)
                                   (sub_int (cvy t.y) radius)
;;

let do_forward dist draw =
  let x1 = t.x + dist * cos (cvh t.heading)
  and y1 = t.y + dist * sin (cvh t.heading)
  in if draw
     then lineto (cvx x1) (cvy y1)
     else moveto (cvx x1) (cvy y1);
     t.x <- x1;
     t.y <- y1
;;
let do_turn a = t.heading <- mod360 (t.heading + a);;

let draw_turtle () =
  let x = t.x
  and y = t.y
  and heading = t.heading
  in set_color foreground;
     set_line_width 1;
     do_turn 90.0;
     do_forward 5.0 true;
     do_turn (-120.0);
     do_forward 10.0 true;
     do_turn (-120.0);
     do_forward 10.0 true;
     do_turn (-120.0);
     do_forward 5.0 true;
     do_turn (-90.0);
     do_forward 2.0 true;
     moveto (cvx x) (cvy y);
     t.x <- x;
     t.y <- y;
     set_color t.color;
     set_line_width t.width;
     t.heading <- heading
;;

let end_drawing () =
  if t.shown
  then (blit_image saved_background (sub_int (cvx t.x) radius)
                                    (sub_int (cvy t.y) radius);
        draw_turtle ())
;;

let forward dist =
  begin_drawing ();
  do_forward dist t.down;
  end_drawing ()
;;

let backward dist = forward (-dist);;

let show () = begin_drawing (); t.shown <- true; end_drawing ();;
let hide () = begin_drawing (); t.shown <- false; end_drawing ();;

let right a = begin_drawing (); do_turn a; end_drawing ();;
let left a = right (-a);;

let color col = t.color <- col; set_color col;;
let width w = t.width <- w; set_line_width w;;

let pen_up () = t.down <- false;;
let pen_down () = t.down <- true;;
let pen_rub () = t.down <- true; color background;;
let pen_draw () = t.down <- true; color foreground;;

let get_state () =
  (t.x, t.y, t.heading, t.down, t.color, t.width, t.shown)
;;
let do_set_state u =
  t.x <- u.x;
  t.y <- u.y;
  moveto (cvx t.x) (cvy t.y);
  t.heading <- mod360 u.heading;
  t.down <- u.down;
  t.color <- u.color;
  set_color t.color;
  t.width <- u.width;
  set_line_width t.width;
  t.shown <- u.shown
;;

let set_state (x, y, heading, down, color, width, shown) =
  begin_drawing ();
  do_set_state {x = x; y = y; heading = heading;
                down = down; color = color; width = width; shown = shown};
  end_drawing ()
;;

let init () =
  xcenter := div_int (size_x ()) 2;
  ycenter := div_int (size_y ()) 2;
  set_color background;
  fill_rect 0 0 (size_x ()) (size_y ());
  do_set_state { x = 0.0; y = 0.0; heading = 0.0;
                 down = true; color = foreground; width = 1; shown = true };
  end_drawing ()
;;

#close "float";;
