(* To execute toplevel phrases *)

#open "meta";;
#open "misc";;
#open "const";;
#open "lexer";;
#open "parser";;
#open "modules";;
#open "location";;
#open "syntax";;
#open "typing";;
#open "ty_decl";;
#open "front";;
#open "back";;
#open "pr_type";;
#open "pr_value";;
#open "symtable";;
#open "load_phr";;

(* Parsing functions *)

let parse_phrase parsing_fun lexing_fun lexbuf =
  let rec skip () =
    try
      match lexing_fun lexbuf with
        EOF -> ()
      | SEMISEMI -> ()
      | _ -> skip()
    with lexer__Lexical_error(_,_,_) ->
      skip() in
  try
    parsing_fun lexing_fun lexbuf
  with parsing__Parse_error f ->
         let pos1 = lexing__get_lexeme_start lexbuf in
         let pos2 = lexing__get_lexeme_end lexbuf in
         if f (obj__repr EOF) or f (obj__repr SEMISEMI) then () else skip();
         prerr_location (Loc(pos1, pos2));
         prerr_begline " Syntax error.";
         prerr_endline "";
         raise Toplevel
     | lexer__Lexical_error(msg, pos1, pos2) ->
         if pos1 >= 0 & pos2 >= 0 then prerr_location (Loc(pos1, pos2));
         prerr_begline " Lexical error: ";
         prerr_string msg;
         prerr_endline ".";
         skip();
         raise Toplevel
     | Toplevel ->
         skip ();
         raise Toplevel
;;

let parse_impl_phrase = parse_phrase Implementation Main
and parse_intf_phrase = parse_phrase Interface Main
;;

(* Execution of directives *)

let do_open name =
  used_modules := find_module name :: !used_modules
;;

let do_directive = function
    Zdir("open", name) ->
      do_open name
  | Zdir("close", name) ->
      used_modules := exceptq (find_module name) !used_modules
  | Zdir("infix", name) ->
      add_infix name
  | Zdir("uninfix", name) ->
      remove_infix name
  | Zdir("directory", dirname) ->
      load_path := dirname :: !load_path
  | Zdir(d, name) ->
      prerr_begline "> Unknown directive \"";
	  prerr_string d;
      prerr_endline2 "\"."
;;

(* Executing phrases *)

let do_toplevel_phrase (Impl(desc,loc)) =
  reset_type_expression_vars [];
  begin match desc with
    Zexpr expr ->
      let ty =
        type_expression loc expr in
      let res =
        load_phrase(compile_lambda false (translate_expression expr)) in
      print_begline "- : ";
      print_one_type ty;
      print_string " = ";
      print_value res ty;
      print_endline ""
  | Zletdef(rec_flag, pat_expr_list) ->
      let env = type_letdef loc rec_flag pat_expr_list in
      if rec_flag then
        load_phrase
          (compile_lambda true (translate_letdef_rec loc pat_expr_list))
      else
        load_phrase
          (compile_lambda false(translate_letdef loc pat_expr_list));
      reset_rollback ();
      do_list
        (fun (name, (typ, mut_flag)) ->
          print_begline name; print_string " : ";
          print_one_type typ; print_string " = ";
          print_value
            global_data.(get_slot_for_variable
                         {qual=compiled_module_name(); id=name})
            typ;
          print_endline "")
        env
  | Ztypedef decl ->
      type_typedecl loc decl;
      do_list
        (fun (name, _, _) ->
            print_begline "Type ";
            print_string name;
            print_endline " defined.")
        decl
  | Zexcdef decl ->
      type_excdecl loc decl;
      do_list
        (fun decl ->
            print_begline "Exception ";
            print_string(match decl with Zconstr0decl name -> name
                                       | Zconstr1decl(name,_,_) -> name);
            print_endline " defined.")
        decl
  | Zimpldirective dir ->
      do_directive dir
  end;
  flush std_out
;;
