(* System functions for interactive use. *)

#open "obj";;
#open "meta";;
#open "misc";;
#open "location";;
#open "lexer";;
#open "modules";;
#open "globals";;
#open "types";;
#open "instruct";;
#open "patch";;
#open "emit_phr";;
#open "symtable";;
#open "do_phr";;
#open "load_phr";;
#open "pr_value";;

let add_suffix name suffix =
  if filename__check_suffix name suffix
  then (filename__chop_suffix name suffix, name)
  else (name, name ^ suffix)
;;

(* Loading in core a compiled bytecode file *)

let load_object name =
  let (_, filename) = add_suffix name ".zo" in
  let truename = find_in_path filename in
  let inchan = open_in_bin truename in
  let stop = input_binary_int inchan in
  let start = pos_in inchan in
  let code_len = stop - start in
  let block_len = code_len + 1 in
  let code = static_alloc block_len in
  fast_really_input inchan code 0 code_len;
  set_nth_char code code_len (char_of_int opcodes__STOP);
  let phrase_index = (input_value inchan : compiled_phrase list) in
  close_in inchan;
  do_list
    (function phr ->
      patch_object code (phr.cph_pos - start) phr.cph_reloc)
    (rev phrase_index);
  do_code false code 0 block_len;
  ()
;;

(* Loading an ML source file *)

let loadfile filename =
  let truename = find_in_path filename in
  let ic = open_in truename in
  let lexbuf = lexing__create_lexer_channel ic in
  let saved_input_name = !input_name
  and saved_input_chan = !input_chan
  and saved_input_lexbuf = !input_lexbuf in
    input_name := truename;
    input_chan := ic;
    input_lexbuf := lexbuf;
    try
      while true do
        do_toplevel_phrase(parse_impl_phrase lexbuf)
      done
    with x ->
      input_lexbuf := saved_input_lexbuf;
      input_chan := saved_input_chan;
      close_in ic;  (* Order changed to avoid race conditions. DD *)
      input_name := saved_input_name;
      match x with
        End_of_file -> ()
      | Toplevel -> ()
      | _ -> raise x
;;

let include name =
  let (simplename, filename) = add_suffix name ".ml" in
    loadfile filename
;;

let load name =
  let (simplename, filename) = add_suffix name ".ml" in
  let modname = filename__basename simplename in
  let saved_defined_module = !defined_module
  and saved_used_modules = !used_modules
  and saved_type_stamp = !type_stamp in
  try
    new_globalenv modname;
    loadfile filename;
    defined_module := saved_defined_module;
    used_modules := saved_used_modules;
    type_stamp := saved_type_stamp;
    ()
  with x ->
    defined_module := saved_defined_module;
    used_modules := saved_used_modules;
    type_stamp := saved_type_stamp;
    raise x
;;

(* To quit. (Alternative: ctrl-D) *)

let quit x = io__exit 0; ()
;;

(* The trace *)

let trace_env = ref ([] : (int * obj) list);;

let rec trace_instr name obj ty =
  match (type_repr ty).typ_desc with
    Tarrow(t1,t2) ->
      let namestar = name ^ "*" in
      repr(fun arg ->
        print_begline name; print_string " <-- ";
        print_value arg t1; print_endline "";
        try
          let res = (magic_obj obj : obj -> obj) arg in
            print_begline name; print_string " --> ";
            print_value res t2; print_endline "";
            trace_instr namestar res t2
        with exc ->
          print_begline name; print_string " raises exception ";
          print_value (repr exc) builtins__type_exn; print_endline "";
          raise exc)
  | _ -> obj
;;

let trace name =
  try
    let val_desc = find_value_desc (GRname name) in
    let pos = get_slot_for_variable val_desc.qualid in
      if mem_assoc pos !trace_env then begin
        prerr_begline "> "; prerr_string name;
        prerr_endline " is already traced."
      end else begin
        trace_env := (pos, global_data.(pos)) :: !trace_env;
        global_data.(pos) <-
          trace_instr name global_data.(pos) val_desc.info.val_typ;
        prerr_begline "> "; prerr_string name;
        prerr_endline " is now traced."
      end
  with Desc_not_found ->
    prerr_begline ">> "; prerr_string name; prerr_endline " is undefined."
;;

let untrace name =
  try
    let val_desc = find_value_desc (GRname name) in
    let pos = get_slot_for_variable val_desc.qualid in
    let rec except = function
      [] -> prerr_begline "> "; prerr_string name;
            prerr_endline " was not traced.";
            []
    | (pos',obj as pair)::rest ->
        if pos == pos' then begin
          global_data.(pos) <- obj;
          prerr_begline "> "; prerr_string name;
          prerr_endline " is no more traced.";
          rest
        end else
          pair :: except rest
    in
      trace_env := except !trace_env;
      ()
  with Desc_not_found ->
    prerr_begline ">> "; prerr_string name; prerr_endline " is undefined."
;;

(* To define specific printing functions. *)

let new_printer name printer =
  try
    let typ_desc =
      find_type_desc (GRname name) in
    printers := (typ_desc.info.ty_constr, magic printer) :: !printers;
    ()
  with Not_found ->
    prerr_begline ">> Type "; prerr_string name;
    prerr_endline " is undefined."
;;

let default_printer name =
  try
    let ty_constr =
      (find_type_desc (GRname name)).info.ty_constr in
    let rec remove = function
      [] -> []
    | (ty,_ as pair)::rest ->
        if same_type_constr ty ty_constr then rest else pair :: remove rest in
    printers := remove !printers;
    ()
  with Not_found ->
    prerr_begline ">> Type "; prerr_string name;
    prerr_endline " is undefined."
;;

let gc () = meta__gc ();;

let cd s = sys__chdir s;;
