/****************************************************************************/
/* PQ.C --- Support Routines for PRINT.COM Multiplex Interrupt              */
/****************************************************************************/
/* These routines provide support for the print queue multiplex interrupt.  */
/* They are for use with Microsoft C v4.0.  The routines are:               */
/*                                                                          */
/* PrintInstalled --- See if PRINT.COM is installed (Fn 0).                 */
/* PrintSubmit    --- Submit a file to the print queue (Fn 1).              */
/* PrintCancel    --- Cancel file(s) from the print queue (Fn 2).           */
/* PrintCancelAll --- Cancel all files from the print queue (Fn 3).         */
/* PrintStatus    --- Freeze queue and get pointer to queue list (Fn 4).    */
/* PrintStatusEnd --- Unfreeze queue.                                       */
/*                                                                          */
/* Notes:                                                                   */
/*    1.  Compile with /Zp switch.  Otherwise, the struct "packet" in       */
/*        PrintSubmit will have an extra byte added between "level" and     */
/*        "file".                                                           */
/****************************************************************************/

#include <dos.h>
#include <stdlib.h>


/****************************************************************************/
/* PrintInstalled --- See if PRINT.COM is installed (Fn 0).                 */
/*--------------------------------------------------------------------------*/
/*    This routine checks to see if PRINT.COM is Installed by calling the   */
/* Multiplex Interrupt (0x2F).  If it is, TRUE is returned; otherwise,      */
/* FALSE is returned.                                                       */
/*    PRINT.COM is multiplex number 1.  The Get Installed State is function */
/* number 0.  After the interrupt, AL is 0xFF if PRINT.COM is Installed.    */
/*                                                                          */
/* Parameters:                                                              */
/*    None.                                                                 */
/*                                                                          */
/* Returns:                                                                 */
/*    TRUE or FALSE indicating if PRINT.COM is Installed.                   */
/****************************************************************************/

int PrintInstalled ()
{
	union REGS regs;

	regs.x.ax = 0x0100;
	int86 (0x2F, &regs, &regs);
	return (regs.h.al == 0xFF);

} /* end PrintInstalled */;


/****************************************************************************/
/* PrintSubmit    --- Submit a file to the print queue (Fn 1).              */
/*--------------------------------------------------------------------------*/
/*    Queue a file to be printed.  Return the error code.  The file must    */
/* consist of a drive designator, full path, and full name.  file names     */
/* containing wild card characters are not allowed.                         */
/*                                                                          */
/* Parameters:                                                              */
/*    File --- Pointer to string containing name to be queued.              */
/*                                                                          */
/* Returns:                                                                 */
/*    Status:  0 == No error, file queued.                                  */
/*             2 == File not found.                                         */
/*             3 == Path not found.                                         */
/*             4 == Too many open files.                                    */
/*             5 == Access Denied.                                          */
/*             8 == Queue Full.                                             */
/*             9 == Busy.                                                   */
/*            12 == Name too long.                                          */
/*            15 == Invalid Drive.                                          */
/****************************************************************************/

int PrintSubmit (File)
char *File;
{
	union REGS regs;
	struct {
		char level;
		char far *file;
	} packet;

	packet.level = 0;
	packet.file  = (char far *) File;
	regs.x.ax = 0x0101;
	regs.x.dx = (unsigned) &packet;
	int86 (0x2F, &regs, &regs);
	return (regs.x.ax > 15) ? (0) : (regs.x.ax);

} /* end PrintSubmit */


/****************************************************************************/
/* PrintCancel    --- Cancel file(s) from the print queue (Fn 2).           */
/*--------------------------------------------------------------------------*/
/*    Cancel one or more files from the queue.  File names containing wild  */
/* card characters are allowed.                                             */
/*                                                                          */
/* Parameters:                                                              */
/*    File --- Pointer to string containing name to be queued.              */
/*                                                                          */
/* Returns:                                                                 */
/*    Nothing.                                                              */
/****************************************************************************/

void PrintCancel (File)
char *File;
{
	union REGS regs;

	regs.x.ax = 0x0102;
	regs.x.dx = (unsigned) File;
	int86 (0x2F, &regs, &regs);

} /* end PrintCancel */


/****************************************************************************/
/* PrintCancelAll --- Cancel all files from the print queue (Fn 3).         */
/*--------------------------------------------------------------------------*/
/*    Cancel printing of all files currently in the print queue.            */
/*                                                                          */
/* Parameters:                                                              */
/*    None.                                                                 */
/*                                                                          */
/* Returns:                                                                 */
/*    Nothing.                                                              */
/****************************************************************************/

void PrintCancelAll ()
{
	union REGS regs;

	regs.x.ax = 0x0103;
	int86 (0x2F, &regs, &regs);

} /* end PrintCancelAll */


/****************************************************************************/
/* PrintStatus    --- Freeze Queue and Get Status (Fn 4).                   */
/*--------------------------------------------------------------------------*/
/*    Freeze the print queue and return a pointer to the queue list and the */
/* number of errors encountered trying to write the last char to the print  */
/* device.                                                                  */
/*                                                                          */
/* Parameters:                                                              */
/*    ErrCnt   --- Pointer to an integer in which to store error count.     */
/*    QueuePtr --- Pointer to a far pointer to char in which to store the   */
/*                 address of the print queue.                              */
/*                                                                          */
/* Returns:                                                                 */
/*    Nothing.                                                              */
/****************************************************************************/

void PrintStatus (QueuePtr, ErrCnt)
char     *far*QueuePtr;
unsigned *ErrCnt;
{
	union  REGS  regs;
	struct SREGS sregs;

	regs.x.ax = 0x0104;
	int86x (0x2F, &regs, &regs, &sregs);
	*ErrCnt   = regs.x.dx;
	FP_SEG(*QueuePtr) = sregs.ds;
	FP_OFF(*QueuePtr) = regs.x.si;

} /* end PrintStatusEnd */


/****************************************************************************/
/* PrintStatusEnd --- Unfreeze queue and get error code (Fn 5).             */
/*--------------------------------------------------------------------------*/
/*    Release the queue from PrintStatus (Fn 4) and return the status.      */
/*                                                                          */
/* Parameters:                                                              */
/*    None.                                                                 */
/*                                                                          */
/* Returns:                                                                 */
/*    Nothing.                                                              */
/****************************************************************************/

void PrintStatusEnd ()
{
	union REGS regs;

	regs.x.ax = 0x0105;
	int86 (0x2F, &regs, &regs);

} /* end PrintStatusEnd */
