program REFLECT;
uses crt;

{ Read an existing SOLMODL data file, and reflect all the data across
  the X axis. }

const MAXNODES = 1000;      { maximum # of nodes in the entire solid }
      MAXCONNECT = 3000;    { maximum # of connections in entire solid }
      MAXSURF = 1000;       { maximum # of surfaces in entire solid }
                            { (MAXSURF = MAXCONNECT / 3) }
      MAXMATL = 30;         { maximum # of materials in entire solid }
      MAXVAR = 20;          { maximum # of numeric inputs on a line }
      MAXLITE = 20;         { maximum # of light sources }

type  text80 = string[80];
      vartype = array[1..MAXVAR] of real;
      nodearray = array[1..MAXNODES] of real;

var   Xworld, Yworld, Zworld: nodearray;
      { world coordinates of each node }
      Connect: array[1..MAXCONNECT] of integer;
      { surface connectivity data }
      Nvert: array[1..MAXSURF] of integer;
      { # vertices per surface }
      Matl: array[1..MAXSURF] of integer;
      { material number of each surface }
      R1, R2, R3: array[1..MAXMATL] of real;
      { material reflectivity constants }
      Color: array[1..MAXMATL] of integer;
      { material color number }
      Ambient: array[1..MAXMATL] of real;
      { ambient light intensity }
      Xlite, Ylite, Zlite: array[1..MAXLITE] of real;
      { coords of light sources }
      Intensity: array[1..MAXLITE] of real;
      { light source intensities }

      Flpurpose: string[127];              { title for plot }
      Maxvert: integer;                    { max # vertices per surface }
      Nsurf: integer;                      { # surfaces }
      Nnodes: integer;                     { # nodes }
      Nlite: integer;                      { # light sources }
      Nmatl: integer;                      { number of materials }
      Nsides: integer;                     { #sides of surface used (1 or 2)}
      Fileread: boolean;                   { flag that a file was read }

{ Two important functions for decoding the Connect array: }

function KONNEC (Surf, Vert: integer): integer;
{ Decode the Connect array to yield the connection data: Vertex Vert of
surface Surf. This function returns an index to the global Xtran, Ytran,
and Ztran arrays (i.e., a node number) }
begin
  Konnec := Connect[(Surf-1) * Maxvert + Vert];
end; { function Konnec }

{ null procedure READINI }
procedure READINI (Flnm: text80);
begin
end;

{ Procedure include files }
{$I INREAL.INC }                         { procedure INREAL }
{$I READFILE.INC }                       { procedure READFILE }

{ variables local to the main procedure }

var Outfile: text;
    Mat: integer;
    Node: integer;
    Surf: integer;
    Vert: integer;

begin
  Fileread := FALSE;
  writeln ('Reading file ', paramstr(1));
  readfile (paramstr(1));
  assign (Outfile, paramstr(2));
  writeln ('Writing file ', paramstr(2));
  rewrite (Outfile);
  writeln (Outfile, Flpurpose);
  writeln (Outfile, 4);
  writeln (Outfile, Nmatl:3, Nnodes*2:5, Nsurf*2:5, Maxvert:3, Nsides:3);
  for Mat := 1 to Nmatl do
    writeln (Outfile, R1[Mat]:7:3, R2[Mat]:7:3, R3[Mat]:7:3, ' ',Color[Mat]:3,
             ' ', Ambient[Mat]:7:3);
  for Node := 1 to Nnodes do
    writeln (Outfile, Xworld[Node]:7:3, Yworld[Node]:7:3, Zworld[Node]:7:3);

{ Reflecting Nodes }
  for Node := 1 to Nnodes do
    writeln (Outfile, -Xworld[Node]:7:3, Yworld[Node]:7:3, Zworld[Node]:7:3);
  for Surf := 1 to Nsurf do begin
    write (Outfile, Nvert[Surf]:3, Matl[Surf]:3);
    for Vert := 1 to Nvert[Surf] do
      write (Outfile, konnec (Surf, Vert):5);
    writeln (Outfile);
  end;
{ Reflecting Surfaces }
  for Surf := 1 to Nsurf do begin
    write (Outfile, Nvert[Surf]:3, Matl[Surf]:3);
    for Vert := 1 to Nvert[Surf] do begin
      Node := konnec (Surf, Nvert[Surf]-Vert+1);
{ If node has X=0 then use the old node number instead of the new.
  Makes interpolation across X=0 possible. }
      if (Xworld[Node] <> 0.0) then
        write (Outfile, (Node + Nnodes):5)
      else
        write (Outfile, Node:5);
    end;
    writeln (Outfile);
  end;
  close (Outfile);
end. { Program Reflect }
