;--------------------------------------------------------------------------
;  MODULE :        XMM                           INITIAL : 19891127 v 1.00
;  AUTHOR :        Jeroen W. Pluimers            UPDATE :  19920705 v 1.20
;
;  DESCRIPTION :   eXtended Memory Manager
;
;  HISTORY :       19891127 - 1.00 - jwp
;
;                      initial translation from XMS 2.0 specification
;
;                  19909322 - 1.11 - jwp
;
;                      final XMS 2.0 implementation
;
;                  19920705 - 1.20 - jwp
;
;                      incorporation of XMS 3.0 specification
;
;                      incorporation of XMS 3.0 specification
;                      compiled for Turbo Pascal 6.0
;                      documentation fixup for XMS 3.0
;
;  COMPUTER :      NEAT-AT, ERC 386/25
;  COMPILER :      TASM 1.0, TASM 2.01
;  LINKER :        TURBO PASCAL 5.0, 5.5 and 6.0
;
;  COPYRIGHT :     (c) 1989-1992 Pluimers Software Ontwikkeling.
;--------------------------------------------------------------------------
                Ideal
                Masm51

                %title  "XMM - eXtended Memory Manager"
                %pagesize 70,132

                Model   TPascal

                DataSeg
;----------------------------------------------------------------------------
; there is nothing in the data segment.
;----------------------------------------------------------------------------

                CodeSeg

;----------------------------------------------------------------------------
; Macro to convert from XMS success/fail to
; a form more acceptable for TURBO PASCAL.  IE.
;
; AX =  1 becomes AL = 0
; AX <> 1 becomes AL = BL
;----------------------------------------------------------------------------
Macro       SuccessFail
Local       Success

	dec	ax
	jz	Success
        mov     al, bl
Success:

EndM




;----------------------------------------------------------------------------
; Initialised data must reside in the code segment.
;----------------------------------------------------------------------------

XMM_Initialised dw      0

; XMM_Control points to a routine that returns XMMNotInitialised (80h)
; for each XMM call until XMMInstalled is called.

label   XMM_Control     DWord
                dw      OFFSET XMM_NotInitialised
                dw      Code




;---------------------------------------------------------------------------
;  ROUTINE :     XMM_NotInitialised
;
;  DESCRIPTION : Routine called for each XMM function call, when the XMM
;                is not initialised with a call to XMMInstalled.
;
;  RETURNS :     XMMNotInitialised.
;---------------------------------------------------------------------------}
Proc   XMM_NotInitialised FAR

            xor     ax, ax                          ; Immediate failure
            mov     bl, 80h                         ; Not Implemented

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMInstalled
;
;  DESCRIPTION : THIS ROUTINE MUST BE CALLED BEFORE ANY OTHER ROUTINE
;                OR ALL THE OTHER ROUTINES WILL FAIL WITH ERROR CODE $80.
;
;  RETURNS :     False - No XMM driver found.
;                True  - An XMM driver has been found.
;
;  POST :        Internal pointer to XMM driver is established.
;
;  PASCAL :      Function XMMInstalled : Boolean;
;---------------------------------------------------------------------------
Proc    XMMInstalled    FAR
Public  XMMInstalled

            cmp     [XMM_Initialised], 0
            jne     @@Already_Initialised
            mov     ax, 4300h                       ; Test for XMM
            int     2fh
            cmp     al, 80h
            jne     @@NoDriver

            mov     ax, 4310h                       ; Get Control Function
            int     2fh
            mov     [word ptr cs:XMM_Control], bx
            mov     [word ptr cs:XMM_Control+2], es
            inc     [XMM_Initialised]

    @@NoDriver:

    @@Already_Initialised:

            mov     ax, [XMM_Initialised]

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMVersion
;
;  DESCRIPTION : Get version numbers. Both the XMS version number and the
;                driver internal revision number are encoded as follows :
;                  Low byte  = minor part.
;                  High byte = major part.
;                e.g. XMSversion $0277 would mean version 2.77.
;                The HMAAvailable indicates the existence of the HMA (not
;                its availability) and is intended mainly for installation
;                programs.
;
;  OUT :         XMSversion   - XMS version number.
;                XMSrevision  - Driver internal revision number.
;                HMAAvailable - False - no HMA detected.
;                               True  - HMA has been detected.
;
;  RETURNS :     XMMNotImplemented - XMMInstalled has not been called.
;                XMMOk             - Always if XMMInstalled has been called.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMVersion(Var XMSversion,
;                                        XMSrevision  : Word;
;                                    Var HMAAvailable : Boolean) : Byte;
;
;  XMS :         ARGS:   AH = 00h
;                RETS:   AX = XMS version number (BCD encoded)
;                        BX = Driver internal revision number.
;                        DX = 0001h if the HMA exists, 0000h otherwise.
;                ERRS:   None.
;---------------------------------------------------------------------------}
Proc    XMMVersion  FAR XMSVersion   : DWord,   \
                        Revision     : DWord,   \
                        HMAAvailable : DWord
Public  XMMVersion

            xor     ah,ah                       ; Function 0
            call    [XMM_Control]

            les     di,[XMSVersion]
            mov     [Word Ptr es:di],ax

            les     di,[Revision]
            mov     [Word Ptr es:di],bx

            les     di,[HMAAvailable]
            mov     [Word Ptr es:di],dx

            xor     al,al                       ; return XMMOk

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMRequestHMA
;
;  DESCRIPTION : Attempts to reserve the whole 64K-16 byte high memory
;                area for the caller.
;                If the HMA is currently unused, the caller's size
;                parameter is compared to the /HMAMIN= parameter on
;                the driver's command line.
;                If the value passed by the caller is greater than or
;                equal to the amount specified by the driver's
;                parameter, the request succeeds.
;                This provides the ability to ensure that programs
;                which use the HMA efficiently have priority over
;                those which do not.
;                NOTE: See the sections "Prioritizing HMA Usage" and
;                      "High Memory Area Restrictions" in the
;                      documentation for more information.
;
;  IN :          SpaceNeeded - Number of bytes in the HMA needed by caller.
;                              It is recommended that if the caller is
;                              an application program this value is
;                              set to 65535 and if the caller is a TSR
;                              the real number of bytes is used.
;
;  RETURNS :     XMMOk              - the HMA is assigned to the caller.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMHMANotExist     - the HMA does not exist.
;                XMMHAMInUse        - the HMA is already in use.
;                XMMHAMMinSize      - SpaceNeeded is less than the /HMAMIN=
;                                     environment parameter.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMRequestHMA (SpaceNeeded : Word) : Byte;
;
;  XMS :         ARGS:   AH = 01h
;                        If the caller is a TSR or device driver,
;                            DX = Space needed in the HMA by the caller in
;                                 bytes
;                        If the caller is an application program,
;                            DX = FFFFh
;                RETS:   AX = 0001h if the HMA is assigned to the caller,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = 90h if the HMA does not exist.
;                        BL = 91h if the HMA is already in use.
;                        BL = 92h if DX is less than the /HMAMIN= parameter.
;---------------------------------------------------------------------------
Proc    XMMRequestHMA FAR SpaceNeeded : Word
Public  XMMRequestHMA

            mov     ah,1
            mov     dx,[SpaceNeeded]

            call    [XMM_Control]

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMReleaseHMA
;
;  DESCRIPTION : Releases the high memory area and allows other
;                programs to use it.
;                Programs which allocate the HMA must release it
;                before exiting.
;                When the HMA has been released, any code or data
;                stored in it becomes invalid and should not be
;                accessed.
;
;  RETURNS :     XMMOk              - the HMA is successfully released.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMHMANotExist     - the HMA does not exist.
;                XMMNotAllocated    - the HMA was not allocated.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMReleaseHMA : Byte;
;
;  XMS :         ARGS:   AH = 02h
;                RETS:   AX = 0001h if the HMA is successfully released,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = 90h if the HMA does not exist.
;                        BL = 93h if the HMA was not allocated.
;---------------------------------------------------------------------------
Proc    XMMReleaseHMA FAR
Public  XMMReleaseHMA

            mov     ah,2

            call    [XMM_Control]

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMGlobalEnableA20
;
;  DESCRIPTION : Attempts to enable the A20 line.
;                It should only be used by programs which have control
;                of the HMA.
;                The A20 line should be turned off via
;                XMMGlobalDisableA20 before a program releases control
;                of the system.
;                NOTE: On many machines, toggling the A20 line is a
;                      relatively slow operation.
;
;  RETURNS :     XMMOk              - the A20 line is enabled.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMA20Err          - and A20 error occurs.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMGlobalEnableA20 : Byte;
;
;  XMS :         ARGS:   AH = 03h
;                RETS:   AX = 0001h if the A20 line is enabled,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = 82h if an A20 error occurs.
;---------------------------------------------------------------------------
Proc    XMMGlobalEnableA20 FAR
Public  XMMGlobalEnableA20

            mov     ah,3
            call    [XMM_Control]

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMGlobalDisableA20
;
;  DESCRIPTION : Attempts to disable the A20 line.
;                It should only be used by programs which have control
;                of the HMA.
;                The A20 line should be disabled before a program
;                releases control of the system.
;                NOTE: On many machines, toggling the A20 line is a
;                      relatively slow operation.
;
;  RETURNS :     XMMOk              - the A20 line is disabled.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMA20Err          - and A20 error occurs.
;                XMMA20StillEnabled - the A20 line is still enabled.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMGlobalDisableA20 : Byte;
;
;  XMS :         ARGS:   AH = 04h
;                RETS:   AX = 0001h if the A20 line is disabled,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = 82h if an A20 error occurs.
;                        BL = 94h if the A20 line is still enabled.
;---------------------------------------------------------------------------
Proc    XMMGlobalDisableA20 FAR
Public  XMMGlobalDisableA20

            mov     ah,4
            call    [XMM_Control]

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMLocalEnableA20
;
;  DESCRIPTION : Attempts to enable the A20 line.
;                It should only be used by programs which need direct
;                access to extended memory.
;                Programs which use this function should call
;                XMMLocalDisableA20 before releasing control of the
;                system.
;                NOTE: On many machines, toggling the A20 line is a
;                      relatively slow operation.
;
;  RETURNS :     XMMOk              - the A20 line is enabled.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMA20Err          - and A20 error occurs.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMLocalEnableA20 : Byte;
;
;  XMS :         ARGS:   AH = 05h
;                RETS:   AX = 0001h if the A20 line is enabled,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = 82h if an A20 error occurs.
;---------------------------------------------------------------------------
Proc    XMMLocalEnableA20 FAR
Public  XMMLocalEnableA20

            mov     ah,5
            call    [XMM_Control]


            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMLocalDisableA20
;
;  DESCRIPTION : Cancels a previous call to XMMLocalEnableA20.
;                It should only be used by programs which need direct
;                access to extended memory.
;                Previous calls to XMMLocalEnableA20 must be canceled
;                before releasing control of the system.
;                NOTE: On many machines, toggling the A20 line is a
;                      relatively slow operation.
;
;  RETURNS :     XMMOk              - the A20 line is disabled.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMA20Err          - and A20 error occurs.
;                XMMA20StillEnabled - the A20 line is still enabled.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMLocalDisableA20 : Byte;
;
;  XMS :         ARGS:   AH = 06h
;                RETS:   AX = 0001h if the function succeeds,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = 82h if an A20 error occurs.
;                        BL = 94h if the A20 line is still enabled.
;---------------------------------------------------------------------------
Proc    XMMLocalDisableA20 FAR
Public  XMMLocalDisableA20

            mov     ah,6
            call    [XMM_Control]

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMQueryA20
;
;  DESCRIPTION : Checks to see if the A20 line is physically enabled.
;                It does this in a hardware independent manner by
;                seeing if "memory wrap" occurs.
;
;  OUT :         A20IsOn  - False - the A20 line is not physically enabled.
;                           True  - the A20 line us physically enabled.
;
;  RETURNS :     XMMOk              - the function succeeds.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMQueryA20 (Var A20IsOn : Boolean) : Byte;
;
;  XMS :         ARGS:   AH = 07h
;                RETS:   AX = 0001h if the A20 line is physically enabled,
;                             0000h otherwise.
;                ERRS:   BL = 00h if the function succeeds.
;                        BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;---------------------------------------------------------------------------
Proc    XMMQueryA20 FAR A20IsOn : DWord
Public  XMMQueryA20

            mov     ah,7

            call    [XMM_Control]

            les     di,[A20IsOn]
            mov     [Byte Ptr es:di],al

            mov     al,bl

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMQueryFree
;
;  DESCRIPTION : Returns the size of the largest available extended
;                memory block and total free memory in the system.
;                NOTE: The 64K HMA is not included in the returned
;                      value even if it is not in use.
;
;  OUT :         LargestFree - Size of the largest free extended
;                              memory block in K-bytes.
;                TotalFree   - Total amount of free extended memory in
;                              K-bytes.
;
;  RETURNS :     XMMOk              - the function succeeds.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMOutOfMemory     - all available extended memory is
;                                     allocated.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMQueryFree (Var LargestFree,
;                                           TotalFree : Word): Byte;
;
;  XMS :         ARGS:   AH = 08h
;                RETS:   AX = Size of the largest free extended memory block
;                             in K-bytes.
;                        DX = Total amount of free extended memory in K-bytes.
;                ERRS:   BL = 00h if the function succeeds.
;                        BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A0h if all extended memory is allocated.
;---------------------------------------------------------------------------
Proc    XMMQueryFree FAR    LargestFree : DWord,    \
                            TotalFree   : DWord
Public  XMMQueryFree

            mov     ah,8

            call    [XMM_Control]

            les     di,[LargestFree]
            mov     [Word ptr es:di],ax

            les     di,[TotalFree]
            mov     [Word ptr es:di],dx

            mov     al,bl

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMAllocateEMB
;
;  DESCRIPTION : Attempts to allocate a block of the given size out of
;                the pool of free extended memory.
;                If a block is available, it is reserved for the
;                caller and a 16-bit handle to that block is returned.
;                The handle should be used in all subsequent extended
;                memory calls.
;                If no memory was allocated, the returned handle is
;                null.
;
;                NOTE: Extended memory handles are scarce resources.
;                      Programs should try to allocate as few as
;                      possible at any one time.
;                      When all of a driver's handles are in use, any
;                      free extended memory is unavailable.
;
;  IN :          BlockSize - Amount of extended memory being requested
;                            in K-bytes.
;
;  OUT :         Handle    - 16-bit handle to the allocated block.
;
;  RETURNS :     XMMOk              - the block is allocated.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMOutOfMemory     - all available extended memory is
;                                     allocated.
;                XMMOutOfHandles    - all available extended memory
;                                     handles are in use.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMAllocateEMB (    BlockSize : Word;
;                                         Var Handle    : Word) : Byte;
;
;  XMS :         ARGS:   AH = 09h
;                        DX = Amount of extended memory being requested
;                             in K-bytes.
;                RETS:   AX = 0001h if the block is allocated,
;                             0000h otherwise.
;                        DX = 16-bit handle to the allocated block.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A0h if all available extended memory is
;                                 allocated.
;                        BL = A1h if all available extended memory
;                                 handles are in use.
;---------------------------------------------------------------------------
Proc    XMMAllocateEMB FAR  BlockSize : Word,   \
                            Handle    : DWord
Public  XMMAllocateEMB

            mov     ah,9
            mov     dx,[BlockSize]

            call    [XMM_Control]

            les     di,[Handle]
            mov     [Word ptr es:di],dx

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMFreeEMB
;
;  DESCRIPTION : This function frees a block of extended memory which
;                was previously allocated using XMMAllocateEMB.
;                Programs which allocate extended memory should free
;                their memory blocks before exiting.
;                When an extended memory buffer is freed, its handle
;                and all data stored in it become invalid and should
;                not be accessed.
;
;  IN :          Handle - 16-bit handle to the allocated block which
;                         should be freed.
;
;  RETURNS :     XMMOk              - the block is allocated.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMInvalidHandle   - the handle is invalid.
;                XMMEMBLocked       - the block is locked.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMFreeEMB (Handle : Word) : Byte;
;
;  XMS :         ARGS:   AH = 0Ah
;                        DX = Handle to the allocated block which should
;                             be freed.
;                RETS:   AX = 0001h if the block is successfully freed,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A2h if the handle is invalid.
;                        BL = ABh if the handle is locked.
;---------------------------------------------------------------------------
Proc    XMMFreeEMB FAR  Handle : Word
Public  XMMFreeEMB

            mov     ah,0Ah
            mov     dx,[Handle]

            call    [XMM_Control]


            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMMoveEMB
;
;  DESCRIPTION : This function attempts to transfer a block of data
;                from one location to another.
;                It is primarily intended for moving blocks of data
;                between conventional memory and extended memory,
;                however it can be used for moving blocks within
;                conventional memory and within extended memory.
;
;                NOTE: If SourceHandle is set to 0000h, the
;                      SourceOffset is interpreted as a standard
;                      segment:offset pair which refers to memory that
;                      is directly accessible by the processor.
;                      The segment:offset pair is stored in Intel
;                      DWORD notation.  The same is true for
;                      DestHandle and DestOffset.
;
;                SourceHandle and DestHandle do not have to refer to
;                locked memory blocks.
;
;                Length must be even.  Although not required,
;                WORD-aligned moves can be significantly faster on
;                most machines. DWORD aligned move can be even faster
;                on 80386 machines.
;
;                If the source and destination blocks overlap, only
;                forward moves (i.e. where the source base is greater ????
;                than the destination base) are guaranteed to work
;                properly.
;
;                Programs should not enable the A20 line before
;                calling this function.  The state of the A20 line is
;                preserved.
;
;                This function is guaranteed to provide a reasonable
;                number of interrupt windows during long transfers.
;
;                The MoveStruc is a structure of type ExtMemMoveRec which
;                is defined as follows :
;
;                  ExtMemMoveRec Record Case Boolean of
;                    False :
;                     (Length       : Longint; number of bytes to transfer
;                      SourceHandle : Word;    Handle of source block
;                      SourceOffset : Longint; offset into source
;                      DestHandle   : Word;    Handle of destination block
;                      DestOffset   : Longint; offset into dest block
;                     );
;                   True :
;                     (TheLenght    : Longint; lenght must be even !
;                      ZeroSource   : Word;    zero sourcehandle means a
;                      SourcePtr    : Pointer; ptr to "normal" memory
;                      ZeroDest     : Word;    zero dest handle means a
;                      DestPtr      : Pointer; ptr to "normal" memory
;                     );
;                  End; (* ExtMemMoveRec *)
;
;  IN :          MoveStruc - Pointer to an Extended Memory Move
;                            Structure (see above).
;
;  RETURNS :     XMMOk              - the move is successful.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMA20Err          - an A20 error occurs.
;                XMMSourceHanldeInv - the SourceHandle is invalid.
;                XMMSourceOffsetInv - the SourceOffset is invalid.
;                XMMDestHanleInv    - the DestHandle is invalid.
;                XMMDestOffset      - the DestOffset is invalid.
;                XMMLenInv          - the Length is invalid.
;                XMMOverlap         - the move has an invalid overlap.
;                XMMParity          - a parity error occurs.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMMoveEMB (MoveRec : XMMMoveRec) : Byte;
;
;  XMS :         ARGS:   AH = 0Bh
;                        DS:SI = Pointer to an Extended Memory Move Structure.
;                RETS:   AX = 0001h if the move is successful,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = 82h if an A20 error occurs.
;                        BL = A3h if the SourceHandle is invalid.
;                        BL = A4h if the SourceOffset is invalid.
;                        BL = A5h if the DestHandle is invalid.
;                        BL = A6h if the DestOffset is invalid.
;                        BL = A7h if the Length is invalid.
;                        BL = A8h if the move has an invalid overlap.
;                        BL = A9h if a parity error occurs.
;---------------------------------------------------------------------------
Struc   XMMMoveRec
    Length              dd  ?   ; 32-bit number of bytes to transfer
    SourceHandle        dw  ?   ; Handle of source block
    SourceOffset        dd  ?   ; 32-bit offset into source
    DestHandle          dw  ?   ; Handle of destination block
    DestOffset          dd  ?   ; 32-bit offset into destination block
EndS

Proc    XMMMoveEMB FAR  MoveRec : XMMMoveRec
Public  XMMMoveEMB

            push    ds

            mov     ah,0Bh
            push    ss
            pop     ds
            lea     si,[MoveRec]                ; DS:SI => Description

            call    [XMM_Control]

            SuccessFail

            pop     ds

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMLockEMB
;
;  DESCRIPTION : Locks an extended memory block and returns its base
;                address as a 32-bit linear address.
;                Locked memory blocks are guaranteed not to move.
;                The 32-bit pointer is only valid while the block is
;                locked because unlocked blocks may be moved by the
;                XMS driver.
;                Locked blocks should be unlocked as soon as possible.
;
;                NOTE: A block does not have to be locked before using
;                      XMMMoveEMB.
;
;                "Lock counts" are maintained for EMBs.
;
;  IN :          Handle - Extended memory block handle to lock.
;
;  OUT :         Linear - 32-bit linear address of the locked block.
;
;  RETURNS :     XMMOk              - the block is locked.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMInvalidHandle   - the handle is invalid.
;                XMMLockOverflow    - the block's lock count overflows.
;                XMMLockFail        - the lock fails.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMLockEMB (    Handle : Word;
;                                     Var Linear : Longint) : Byte;
;
;  XMS :         ARGS:   AH = 0Ch
;                        DX = Extended memory block handle to lock
;                RETS:   AX = 0001h if the block is locked,
;                             0000h otherwise.
;                        DX:BX = 32-bit linear address of the locked block.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A2h if the handle is invalid.
;                        BL = ACh if the block's lock count overflows.
;                        BL = ADh if the lock fails.
;---------------------------------------------------------------------------
Proc    XMMLockEMB FAR  Handle : Word,  \
                        Linear : DWord
Public  XMMLockEMB

            mov     ah,0Ch
            mov     dx,[Handle]

            call    [XMM_Control]

            les     di,[Linear]
            mov     [Word ptr es:di],bx         ; low word
            mov     [word ptr es:di+2],dx       ; high word

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMUnLockEMB
;
;  DESCRIPTION : Unlocks a locked extended memory block.  Any 32-bit
;                pointers into the block become invalid and should no
;                longer be used.
;
;  IN :          Handle - Extended memory block handle to unlock.
;
;  RETURNS :     XMMOk              - the block is unlocked.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMInvalidHandle   - the handle is invalid.
;                XMMEMBUnlocked     - the block is not locked.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMUnLockEMB (Handle : Word) : Byte;
;
;  XMS :         ARGS:   AH = 0Dh
;                        DX = Extended memory block handle to unlock.
;                RETS:   AX = 0001h if the block is unlocked,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A2h if the handle is invalid.
;                        BL = AAh if the block is not locked.
;---------------------------------------------------------------------------
Proc    XMMUnLockEMB FAR    Handle : Word
Public  XMMUnLockEMB

            mov     ah,0Dh
            mov     dx,[Handle]

            call    [XMM_Control]

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMGetHandleInfo
;
;  DESCRIPTION : Returns additional information about an extended
;                memory block to the caller.
;
;                NOTE: To get the block's base address, use XMMLockEMB.
;
;  IN :          Handle      - Extended memory block handle.
;
;  OUT :         LockCount   - The block's lock count.
;                FreeHandles - Number of free EMB handles in the system.
;                BlockLength - The block's length in K-bytes.
;
;
;  RETURNS :     XMMOk              - the block's information is found.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMInvalidHandle   - the handle is invalid.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMGetHandleInfo (    Handle      : Word;
;                                           Var LockCount,
;                                               FreeHandles : Byte;
;                                           Var BlockLength : Word) : Byte;
;
;  XMS :         ARGS:   AH = 0Eh
;                        DX = Extended memory block handle
;                RETS:   AX = 0001h if the block's information is found,
;                             0000h otherwise.
;                        BH = The block's lock count.
;                        BL = Number of free EMB handles in the system.
;                        DX = The block's length in K-bytes.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A2h if the handle is invalid.
;---------------------------------------------------------------------------
Proc    XMMGetHandleInfo FAR    Handle      : Word,     \
                                LockCount   : DWord,    \
                                FreeHandles : DWord,    \
                                BlockLength : DWord
Public  XMMGetHandleInfo

            mov     ah,0Eh
            mov     dx,[Handle]

            call    [XMM_Control]

            les     di,[LockCount]
            mov     [Byte ptr es:di],bh

            les     di,[FreeHandles]
            mov     [Byte ptr es:di],bl

            les     di,[BlockLength]
            mov     [Word ptr es:di],dx

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMReallocateEMB
;
;  DESCRIPTION : Attempts to reallocate an unlocked extended memory
;                block so that it becomes the newly specified size.
;                If the new size is smaller than the old block's size,
;                all data at the upper end of the old block is lost.
;
;  IN :          Handle  - Unlocked extended memory block handle to
;                          reallocate.
;                NewSize - New size for the extended memory block in
;                          K-bytes.
;
;  RETURNS :     XMMOk              - the block is reallocated.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMOutOfMemory     - all available extended memory is
;                                     allocated.
;                XMMOutOfHandles    - all available extended memory
;                                     handles are in use.
;                XMMInvalidHandle   - the handle is invalid.
;                XMMEMBLocked       - the block is locked.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMReallocateEMB (Handle,
;                                           NewSize : Word) : Byte;
;
;  XMS :         ARGS:   AH = 0Fh
;                        BX = New size for the extended memory block
;                             in K-bytes.
;                        DX = Unlocked extended memory block handle to
;                             reallocate.
;                RETS:   AX = 0001h if the block is reallocated,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A0h if all available extended memory is
;                                 allocated.
;                        BL = A1h if all available extended memory
;                                 handles are in use.
;                        BL = A2h if the handle is invalid.
;                        BL = ABh if the block is locked.
;---------------------------------------------------------------------------
Proc    XMMReallocateEMB FAR    Handle  : Word, \
                                NewSize : Word
Public  XMMReallocateEMB

            mov     ah,0Fh
            mov     dx,[Handle]
            mov     bx,[NewSize]

            call    [XMM_Control]

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMRequestUMB
;
;  DESCRIPTION : Attempts to allocate an upper memory block to the
;                caller. If the function fails, the size of the
;                largest free UMB is returned in DX.
;
;                NOTE: By definition UMBs are located below the 1MB
;                      address boundary.
;                      The A20 Line does not need to be enabled before
;                      accessing an allocated UMB.
;                      This function does not need to be implemented
;                      by an XMS driver.
;
;                UMBs are paragraph aligned.
;
;                To determine the size of the largest available UMB,
;                attempt to allocate one with a size of FFFFh.
;
;                UMBs are unaffected by EMS calls.
;
;  IN :          RequestSize - Size of requested memory block in
;                              paragraphs (one paragraph is 16 bytes).
;
;  OUT :         UMBSeg      - Segment number of the upper memory block.
;                BlockSize   - If the request is granted, actual size
;                              in paragraphs of the allocated block in
;                              paragraphs.
;                              Otherwise, size of the largest
;                              available UMB in paragraphs.
;
;  RETURNS :     XMMOk              - the request is granted.
;                XMMNotImplemented  - the function is not implemented.
;                XMMUMBSizeTooBig   - a smaller UMB is available.
;                XMMNoUMBs          - no UMBs are available.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMRequestUMB (    RequestSize : Word;
;                                        Var UMBSeg,
;                                            BlockSize   : Word) : Byte;
;
;  XMS :         ARGS:   AH = 10h
;                        DX = Size of requested memory block in paragraphs.
;                RETS:   AX = 0001h if the request is granted,
;                             0000h otherwise.
;                        BX = Segment number of the upper memory block.
;                        If the request is granted,
;                            DX = Actual size of the allocated block
;                                 in paragraphs.
;                        otherwise,
;                            DX = Size of the largest available UMB
;                                 in paragraphs.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = B0h if a smaller UMB is available.
;                        BL = B1h if no UMBs are available.
;---------------------------------------------------------------------------
Proc    XMMRequestUMB FAR   RequestSize : Word,     \
                            UMBSeg      : DWord,    \
                            BlockSize   : DWord
Public  XMMRequestUMB

            mov     ah,10h
            mov     dx,[RequestSize]

            call    [XMM_Control]

            les     di,[UMBSeg]
            mov     [Word ptr es:di],bx

            les     di,[BlockSize]
            mov     [Word ptr es:di],dx

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMReleaseUMB
;
;  DESCRIPTION : This function frees a previously allocated upper
;                memory block.
;                When an UMB has been released, any code or data
;                stored in it becomes invalid and should not be
;                accessed.
;
;                NOTE: This function does not need to be implemented
;                      by an XMS driver.
;
;  IN :          UMBSeg - Segment number of the upper memory block.
;
;  RETURNS :     XMMOk              - the block was released.
;                XMMNotImplemented  - the function is not implemented.
;                XMMInvalidUMB      - the UMB segment number is invalid.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMReleaseUMB (UMBSeg : Word) : Byte;
;
;  XMS :         ARGS:   AH = 11h
;                        DX = Segment number of the upper memory block.
;                RETS:   AX = 0001h if the block was released,
;                             0000h otherwise.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = B2h if the UMB segment number is invalid.
;---------------------------------------------------------------------------
Proc    XMMReleaseUMB FAR   UMBSeg : Word
Public  XMMReleaseUMB

            mov     ah,11h
            mov     dx,[UMBSeg]

            call    [XMM_Control]

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMReallocateUMB
;
;  DESCRIPTION : This function attempts to reallocate an Upper Memory
;                Block to a newly specified size. If the new size is
;                smaller than the old block's size, all data at the
;                upper end of the block is lost.
;
;                NOTE: This function does not need to be implemented
;                      by an XMS driver.
;
;  IN :          UMBSeg - Segment number of the upper memory block.
;
;  RETURNS :     XMMOk              - the block was released.
;                XMMNotImplemented  - the function is not implemented.
;                XMMUMBSizeTooBig   - a smaller UMB is available.
;                XMMInvalidUMB      - the UMB segment number is invalid.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMReallocateUMB (RequestSize : Word;
;                                           UMBSeg : Word) : Byte;
;
;                ARGS:   AH = 12h
;                        BX = New size for UMB in paragraphs
;                        DX = Segment number of the UMB to reallocate
;
;                RETS:   AX = 1 if the block was reallocated, 0 otherwise
;
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = B0h if no UMB large enough to satisfy the
;                                 request is available.
;                                 In this event, DX is returned with the size
;                                 of the largest UMB that is available.
;                        BL = B2h if the UMB segment number is invalid.
;---------------------------------------------------------------------------
Proc    XMMReallocateUMB FAR   RequestSize : Word,     \
                               UMBSeg      : Word
Public  XMMReallocateUMB

            mov     ah,12h
            mov     bx,[RequestSize]
            mov     dx,[UMBSeg]

            call    [XMM_Control]

            SuccessFail

            ret

EndP




P386N  ; code below only works on 386 processors




;---------------------------------------------------------------------------
;  ROUTINE :     XMMQueryAnyFree
;
;  DESCRIPTION : This function uses 32-bit values to return the size of
;                available memory, thus allowing returns up to 4GByte.
;                Additionally, it returns the highest known physical
;                memory address, that is, the physical address of the
;                last byte of memory.  There may be discontinuities in
;                the memory map below this address.
;
;                The memory pool reported on is the same as that
;                reported on by the existing Query Free Extended Memory
;                function 08h.  If the highest memory address is not
;                more than 64 Mb, then these two functions will return
;                the same results.
;
;                Because of its reliance on 32-bit registers, this
;                function is only available on 80386 and higher
;                processors.  XMS drivers on 80286 machines should
;                return error code 80h if this function is called.
;
;                If error code 81h is returned, the value in ECX will
;                still be valid.  If error code A0h is returned, EAX and
;                EDX will be 0, and ECX will still be valid.
;
;  OUT :         LargestFree - Size of the largest free extended
;                              memory block in K-bytes.
;                LastByte    - Physical address of the last byte in memory.
;                TotalFree   - Total amount of free extended memory in
;                              K-bytes.
;
;  RETURNS :     XMMOk              - the function succeeds.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMOutOfMemory     - all available extended memory is
;                                     allocated.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMQueryAnyFree (Var LargestFree,
;                                              LastByte,
;                                              TotalFree : LongInt): Byte;
;
;  XMS :         ARGS:   AH = 88h
;                RETS:   AX = Size of the largest free extended memory block
;                             in K-bytes.
;                        DX = Total amount of free extended memory in K-bytes.
;                RETS:   EAX = Size of largest free extended memory block in Kb.
;                        BL = 0 if no error occurs, otherwise it takes an error code.
;                        ECX = Highest ending address of any memory block.
;                        EDX = Total amount of free memory in Kb.
;
;                ERRS:   BL = 00h if the function succeeds.
;                        BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A0h if all extended memory is allocated.
;---------------------------------------------------------------------------
Proc    XMMQueryAnyFree FAR LargestFree : DWord,    \
                            LastByte    : DWord,    \
                            TotalFree   : DWord
Public  XMMQueryAnyFree

            mov     ah,88h

            call    [XMM_Control]

            les     di,[LargestFree]
            mov     [DWord ptr es:di],eax

            les     di,[LastByte]
            mov     [DWord ptr es:di],ecx

            les     di,[TotalFree]
            mov     [DWord ptr es:di],edx

            mov     al,bl

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMAllocateAnyEMB
;
;  DESCRIPTION : This function is similar to the existing Allocate
;                Extended Memory 09h, except that it uses a 32-bit
;                instead of a 16-bit value to specify the amount of
;                memory requested.  It allocates from the same memory
;                and handle pool as the current function.
;
;                Since it requires a 32-bit register, this function can
;                be supported only on 80386 and higher processors, and
;                XMS drivers on 80286 machines should return error code
;                80h.
;
;  IN :          BlockSize - Amount of extended memory being requested
;                            in K-bytes.
;
;  OUT :         Handle    - 16-bit handle to the allocated block.
;
;  RETURNS :     XMMOk              - the block is allocated.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMOutOfMemory     - all available extended memory is
;                                     allocated.
;                XMMOutOfHandles    - all available extended memory
;                                     handles are in use.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMAllocateAnyEMB (    BlockSize : Longint;
;                                            Var Handle    : Word) : Byte;
;
;  XMS :         ARGS:   AH = 89h
;                        EDX = Amount of extended memory being requested
;                             in K-bytes.
;                RETS:   AX = 0001h if the block is allocated,
;                             0000h otherwise.
;                        DX = 16-bit handle to the allocated block.
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A0h if all available extended memory is
;                                 allocated.
;                        BL = A1h if all available extended memory
;                                 handles are in use.
;---------------------------------------------------------------------------
Proc    XMMAllocateAnyEMB FAR  BlockSize : DWord,   \
                               Handle    : DWord
Public  XMMAllocateAnyEMB

            mov     ah,89h
            mov     edx,[BlockSize]

            call    [XMM_Control]

            les     di,[Handle]
            mov     [Word ptr es:di],dx

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMGetExtendedHandleInfo
;
;  DESCRIPTION : This function is similar to the Get EMB Handle
;                Information function. Since it uses a 32-bit register
;                to report the block size, it can be used to get
;                information on blocks larger than 64 Mb.  It also uses
;                a 16-bit instead of 8-bit register to report the number
;                of free handles, allowing the handle pool to be
;                extended beyond 256 entries.
;
;                Because of its reliance on a 32-bit register, this
;                function is available on 80386 and higher processors.
;                XMS drivers on 80286 machines should return error code
;                80h if this function is called.
;
;  IN :          Handle      - Extended memory block handle.
;
;  OUT :         LockCount   - The block's lock count.
;                FreeHandles - Number of free EMB handles in the system.
;                BlockLength - The block's length in K-bytes.
;
;
;  RETURNS :     XMMOk              - the block's information is found.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMInvalidHandle   - the handle is invalid.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMGetExtendedHandleInfo (    Handle      : Word;
;                                                   Var LockCount   : Byte;
;                                                   Var FreeHandles : Word;
;                                                   Var BlockLength : Longint) : Byte;
;
;  XMS :         ARGS:   AH = 8Eh
;                        DX = Extended memory block handle.
;
;                RETS:   AX = 1 if the block's information is found, 0 if not
;                        BH = Block lock count
;                        CX = Number of free EMB handles in the system
;                        EDX = Block's length in Kb.
;
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A2h if the handle is invalid.
;---------------------------------------------------------------------------
Proc    XMMGetExtendedHandleInfo FAR    Handle      : Word,     \
                                        LockCount   : DWord,    \
                                        FreeHandles : DWord,    \
                                        BlockLength : DWord
Public  XMMGetExtendedHandleInfo

            mov     ah,8Eh
            mov     dx,[Handle]

            call    [XMM_Control]

            les     di,[LockCount]
            mov     [Byte ptr es:di],bh

            les     di,[FreeHandles]
            mov     [Word ptr es:di],cx

            les     di,[BlockLength]
            mov     [DWord ptr es:di],edx

            SuccessFail

            ret

EndP




;---------------------------------------------------------------------------
;  ROUTINE :     XMMReallocateAnyEMB
;
;  DESCRIPTION : This function is similar to the existing Reallocate
;                Extended Memory, except that it uses a 32-bit instead
;                of a 16-bit value to specify the amount of memory
;                requested.  It allocates from the same memory and
;                handle pool as the current function.
;
;                Since it requires a 32-bit register, this function can
;                be supported only on 80386 and higher processors, and
;                XMS drivers on 80286 machines should return error code
;                80h.
;
;  IN :          Handle  - Unlocked extended memory block handle to
;                          reallocate.
;                NewSize - New size for the extended memory block in
;                          K-bytes.
;
;  RETURNS :     XMMOk              - the block is reallocated.
;                XMMNotImplemented  - the function is not implemented.
;                XMMVDiskFound      - a VDISK device is detected.
;                XMMOutOfMemory     - all available extended memory is
;                                     allocated.
;                XMMOutOfHandles    - all available extended memory
;                                     handles are in use.
;                XMMInvalidHandle   - the handle is invalid.
;                XMMEMBLocked       - the block is locked.
;
;  PRE :         XMMInstalled must have been called first.
;
;  PASCAL :      Function XMMReallocateAnyEMB (Handle,
;                                              NewSize : Longint) : Byte;
;
;  XMS :         ARGS:   AH = 8Fh
;                        EBX = New size for extended memory block, in Kb.
;                        DX = Unlocked handle for memory block to be resized.
;
;                RETS:   AX = 1 if the block is reallocated, 0 if not
;
;                ERRS:   BL = 80h if the function is not implemented.
;                        BL = 81h if a VDISK device is detected.
;                        BL = A0h if all available extended memory is allocated.
;                        BL = A1h if all available extended memory handles are in use.
;                        BL = A2h if the handle is invalid.
;                        BL = ABh if the block is locked.
;---------------------------------------------------------------------------
Proc    XMMReallocateAnyEMB FAR    Handle  : Word, \
                                   NewSize : DWord
Public  XMMReallocateAnyEMB

            mov     ah,8Fh
            mov     dx,[Handle]
            mov     ebx,[NewSize]

            call    [XMM_Control]

            SuccessFail

            ret

EndP



    



EndS
End
